// Utility functions for normalizing Egyptian phone numbers

// Valid Egyptian mobile prefixes under +20
const VALID_PREFIXES = ['10', '11', '12', '15'];

function isValidEgyptMobile(msisdn) {
  // msisdn should be digits only after country code: e.g. 10XXXXXXXX, 11XXXXXXXX, 12XXXXXXXX, 15XXXXXXXX
  return /^[1][0-5]\d{8}$/.test(msisdn) && VALID_PREFIXES.includes(msisdn.slice(0, 2));
}

function normalizeEgyptPhone(input) {
  if (!input) return null;

  // Work with digits only
  let raw = String(input).trim();
  let digits = raw.replace(/\D/g, '');

  if (!digits) return null;

  // Handle common international prefixes like 00 and duplicates of 20
  if (digits.startsWith('00')) {
    digits = digits.slice(2);
  }

  // If starts with 2020..., reduce to single 20
  if (digits.startsWith('2020')) {
    digits = '20' + digits.slice(4);
  }

  // If starts with single leading zero for local format, e.g. 01123456789 -> 201123456789
  if (digits.startsWith('0')) {
    digits = '20' + digits.slice(1);
  }

  // If starts directly with mobile msisdn like 11XXXXXXXXX (10 digits), prepend 20
  if (digits.length >= 10 && VALID_PREFIXES.includes(digits.slice(0, 2)) && !digits.startsWith('20')) {
    digits = '20' + digits;
  }

  // Ensure we have country code 20 at the start
  if (!digits.startsWith('20')) {
    const found = extractValidMsisdn(digits);
    return found ? `+20${found}` : null;
  }

  // Remove any additional leading 20 inside the rest
  let rest = digits.slice(2);
  if (rest.startsWith('20')) {
    rest = rest.slice(2);
  }

  // If rest is too long, try to find the correct 10-digit msisdn with valid prefix
  let candidate = null;
  if (isValidEgyptMobile(rest.slice(0, 10))) {
    candidate = rest.slice(0, 10);
  } else {
    candidate = extractValidMsisdn(rest);
  }

  if (candidate) {
    return `+20${candidate}`;
  }

  // As a last attempt, if rest length >= 10 and starts with valid prefix, trim to 10
  if (rest.length >= 10 && VALID_PREFIXES.includes(rest.slice(0, 2))) {
    return `+20${rest.slice(0, 10)}`;
  }

  return null;
}

function extractValidMsisdn(digits) {
  // Search for a 10-digit segment starting with a valid prefix
  for (let i = 0; i <= digits.length - 10; i++) {
    const seg = digits.slice(i, i + 10);
    if (isValidEgyptMobile(seg)) {
      return seg;
    }
  }
  return null;
}

// Export both as default (function) and named to handle different import styles
module.exports = normalizeEgyptPhone;
module.exports.normalizeEgyptPhone = normalizeEgyptPhone;
module.exports.isValidEgyptMobile = isValidEgyptMobile;