const cron = require('node-cron');
const User = require('./models/User');
const { sendEmail } = require('./utils/emailService');

/**
 * Send premium expiry reminders
 * Runs daily at 9:00 AM
 */
const sendExpiryReminders = async () => {
    try {
        console.log('[Premium Expiry] Starting expiry reminder check...');
        
        const now = new Date();
        const sevenDaysFromNow = new Date(now);
        sevenDaysFromNow.setDate(sevenDaysFromNow.getDate() + 7);
        
        // Find users whose premium expires in 7 days
        const usersExpiringSoon = await User.find({
            premiumActive: true,
            premiumExpiry: {
                $gte: now,
                $lte: sevenDaysFromNow
            }
        });

        console.log(`[Premium Expiry] Found ${usersExpiringSoon.length} users expiring in 7 days`);

        for (const user of usersExpiringSoon) {
            try {
                const daysRemaining = Math.ceil((new Date(user.premiumExpiry) - now) / (1000 * 60 * 60 * 24));
                
                const subject = `TaskTree Premium Expiring Soon - ${daysRemaining} Days Remaining`;
                const html = `
                    <h2>Premium Subscription Expiring Soon</h2>
                    <p>Hello ${user.name},</p>
                    <p>Your TaskTree Premium subscription will expire in <strong>${daysRemaining} days</strong>.</p>
                    <p><strong>Expiry Date:</strong> ${new Date(user.premiumExpiry).toLocaleDateString()}</p>
                    <p>To continue enjoying premium features, please renew your subscription.</p>
                    <p><a href="${process.env.FRONTEND_URL || 'https://task-tree.net'}/pricing">Renew Now</a></p>
                    <p>Thank you for using TaskTree!</p>
                `;

                await sendEmail(user.email, subject, html);
                console.log(`[Premium Expiry] Reminder sent to ${user.email}`);
            } catch (emailError) {
                console.error(`[Premium Expiry] Failed to send reminder to ${user.email}:`, emailError.message);
            }
        }

        // Find users whose premium expires today
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        const tomorrow = new Date(today);
        tomorrow.setDate(tomorrow.getDate() + 1);

        const usersExpiringToday = await User.find({
            premiumActive: true,
            premiumExpiry: {
                $gte: today,
                $lt: tomorrow
            }
        });

        console.log(`[Premium Expiry] Found ${usersExpiringToday.length} users expiring today`);

        for (const user of usersExpiringToday) {
            try {
                const subject = 'TaskTree Premium Expiring Today';
                const html = `
                    <h2>Premium Subscription Expiring Today</h2>
                    <p>Hello ${user.name},</p>
                    <p>Your TaskTree Premium subscription expires <strong>today</strong>.</p>
                    <p>To continue enjoying premium features, please renew your subscription immediately.</p>
                    <p><a href="${process.env.FRONTEND_URL || 'https://task-tree.net'}/pricing">Renew Now</a></p>
                    <p>Thank you for using TaskTree!</p>
                `;

                await sendEmail(user.email, subject, html);
                console.log(`[Premium Expiry] Today reminder sent to ${user.email}`);
            } catch (emailError) {
                console.error(`[Premium Expiry] Failed to send today reminder to ${user.email}:`, emailError.message);
            }
        }

        // Auto-downgrade expired subscriptions
        const expiredUsers = await User.find({
            premiumActive: true,
            premiumExpiry: { $lt: now }
        });

        console.log(`[Premium Expiry] Found ${expiredUsers.length} expired subscriptions to downgrade`);

        for (const user of expiredUsers) {
            user.premiumActive = false;
            user.planType = 'free';
            await user.save();
            console.log(`[Premium Expiry] Auto-downgraded user ${user.email}`);
        }

        console.log('[Premium Expiry] Expiry reminder check completed');
    } catch (error) {
        console.error('[Premium Expiry] Error in expiry reminder job:', error);
    }
};

// Schedule daily at 9:00 AM
cron.schedule('0 9 * * *', sendExpiryReminders, {
    scheduled: true,
    timezone: "Africa/Cairo"
});

console.log('✅ Premium expiry reminder cron job scheduled (daily at 9:00 AM)');

module.exports = { sendExpiryReminders };

