const mongoose = require('mongoose');

// Flexible MCQ Choice Schema - supports any number of choices
const MCQChoiceSchema = new mongoose.Schema({
    text: {
        type: String,
        required: true
    },
    isCorrect: {
        type: Boolean,
        default: false
    }
}, { _id: true });

// Question Schema - supports multiple question types
const QuestionSchema = new mongoose.Schema({
    type: {
        type: String,
        enum: ['mcq', 'true-false', 'short-answer', 'essay'],
        default: 'mcq'
    },
    question: {
        type: String,
        required: true
    },
    // For MCQ - flexible number of choices
    choices: [MCQChoiceSchema],
    // For true-false
    correctAnswer: {
        type: Boolean,
        default: null
    },
    // For short-answer and essay
    answerKey: {
        type: String,
        default: null
    },
    points: {
        type: Number,
        default: 1
    },
    explanation: {
        type: String,
        default: null
    }
}, { _id: true });

const assignmentSchema = new mongoose.Schema({
    title: {
        type: String,
        required: true
    },
    description: {
        type: String,
        default: ''
    },
    course: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'Course',
        required: true
    },
    createdBy: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'User',
        required: true
    },
    questions: [QuestionSchema],
    totalPoints: {
        type: Number,
        default: 0
    },
    dueDate: {
        type: Date,
        default: null
    },
    timeLimit: {
        type: Number, // in minutes
        default: null
    },
    attemptsAllowed: {
        type: Number,
        default: 1
    },
    status: {
        type: String,
        enum: ['draft', 'published', 'closed'],
        default: 'draft'
    },
    submissions: [{
        student: {
            type: mongoose.Schema.Types.ObjectId,
            ref: 'User'
        },
        answers: [{
            questionId: mongoose.Schema.Types.ObjectId,
            answer: mongoose.Schema.Types.Mixed, // Can be choice ID, boolean, or string
            isCorrect: Boolean,
            points: Number
        }],
        score: {
            type: Number,
            default: 0
        },
        submittedAt: {
            type: Date,
            default: Date.now
        },
        attemptNumber: {
            type: Number,
            default: 1
        }
    }]
}, { timestamps: true });

// Calculate total points before saving
assignmentSchema.pre('save', function(next) {
    if (this.questions && this.questions.length > 0) {
        this.totalPoints = this.questions.reduce((sum, q) => sum + (q.points || 0), 0);
    }
    next();
});

module.exports = mongoose.model('Assignment', assignmentSchema);

