'use strict';

const cron = require('node-cron');
const axios = require('axios');
const phoneUtils = require('./utils/phoneUtils');
const AutomationReport = require('./models/AutomationReport');

const DAILY_SUMMARY_URL ='https://server.task-tree.net/api/daily-summary';
const WASENDER_URL = process.env.WASENDER_URL || 'https://wasenderapi.com/api/send-message';
const WASENDER_API_TOKEN = process.env.WASENDER_API_TOKEN || '30ddf4326470551c5ff032627761b0736d5981e304cb1c5a853bbb99fddac45d';
const CRON_SCHEDULE ='0 23 * * *'; // 23:00 every day
const CRON_TZ = 'Africa/Cairo'; // Default to Cairo to ensure 23:00 Cairo time
const SEND_DELAY_MS = Number(5000);
const ENABLED = (process.env.DAILY_AUTOMATION_ENABLED ?? 'true').toLowerCase() !== 'false';
const DRY_RUN = (process.env.DAILY_AUTOMATION_DRY_RUN ?? 'false').toLowerCase() === 'true';

const logPrefix = '[DailyAutomation]';

function sleep(ms) { 
  return new Promise(res => setTimeout(res, ms)); 
}

// Helper: normalize Egyptian phone using shared util
function normalizePhone(input) {
  try {
    const normalizer = (typeof phoneUtils === 'function')
      ? phoneUtils
      : (phoneUtils && typeof phoneUtils.normalizeEgyptPhone === 'function')
        ? phoneUtils.normalizeEgyptPhone
        : null;
    return normalizer ? normalizer(input) : String(input || '').trim();
  } catch (_) {
    return String(input || '').trim();
  }
}

function isValidEgyptPhone(to) {
  const digits = String(to || '').replace(/\D/g, '');
  const msisdn = digits.startsWith('20') ? digits.slice(2) : digits;
  try {
    return !!phoneUtils.isValidEgyptMobile && phoneUtils.isValidEgyptMobile(msisdn);
  } catch (_) {
    return true;
  }
}

async function fetchSummaries() {
  console.log(`${logPrefix} Fetching summaries from ${DAILY_SUMMARY_URL} ...`);
  const res = await axios.get(DAILY_SUMMARY_URL, { timeout: 20000 });
  const data = res?.data;
  if (!Array.isArray(data)) {
    throw new Error(`Unexpected response type: ${typeof data}`);
  }
  console.log(`${logPrefix} Received ${data.length} summary item(s).`);
  return data;
}

function normalizeText(text) {
  if (typeof text !== 'string') return '';
  // Strip backticks commonly used for inline formatting or URLs
  return text.replace(/`/g, '').trim();
}

async function sendMessage(to, text) {
  const payload = { to, text: normalizeText(text) };
  if (DRY_RUN) {
    console.log(`${logPrefix} DRY_RUN → would send message:`, payload);
    return { dryRun: true, to };
  }
  console.log(`${logPrefix} Sending message to ${to} ...`);
  try {
    const res = await axios.post(WASENDER_URL, payload, {
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${WASENDER_API_TOKEN}`
      },
      timeout: 20000
    });
    console.log(`${logPrefix} Sent to ${to} (status ${res.status}).`);
    return res.data;
  } catch (err) {
    const status = err.response?.status;
    const body = err.response?.data;
    console.error(`${logPrefix} Error sending to ${to}:`, status ? `HTTP ${status}` : err.message, body ? JSON.stringify(body) : '');
    throw err;
  }
}

function formatErrorReason(err) {
  try {
    const status = err?.response?.status;
    const body = err?.response?.data;
    const msg = body?.message || body?.error || err?.message || 'Unknown error';
    return status ? `HTTP ${status} - ${String(msg)}` : String(msg);
  } catch (_) {
    return err?.message || 'Unknown error';
  }
}

async function sendWithRetries(to, text, maxAttempts = 3) {
  const attemptsLog = [];
  if (DRY_RUN) {
    return { ok: true, attempts: 0, attemptsLog, dryRun: true, data: { dryRun: true, to } };
  }
  for (let attempt = 1; attempt <= maxAttempts; attempt++) {
    try {
      const data = await sendMessage(to, text);
      return { ok: true, attempts: attempt, attemptsLog, data };
    } catch (err) {
      const reason = formatErrorReason(err);
      const code = err?.response?.status || null;
      attemptsLog.push({ attempt, code, reason, timestamp: new Date() });
      const retryAfterSec = Number(err?.response?.data?.retry_after) || 0;
      const waitMs = retryAfterSec ? Math.max(SEND_DELAY_MS, retryAfterSec * 1000) : Math.max(SEND_DELAY_MS, 1000 * attempt);
      if (attempt < maxAttempts) {
        console.log(`${logPrefix} Attempt ${attempt} failed to=${to}: ${reason}. Waiting ${waitMs}ms before retry...`);
        await sleep(waitMs);
      } else {
        return { ok: false, attempts: attempt, attemptsLog, error: err };
      }
    }
  }
}

async function runDailySummaryOnce() {
  const startedAt = new Date();
  console.log(`${logPrefix} Job started at ${startedAt.toISOString()}`);
  try {
    const items = await fetchSummaries();

    if (!items.length) {
      console.log(`${logPrefix} No summaries to process.`);
      return { processed: 0 };
    }

    const run = new AutomationReport({
      type: 'daily',
      runId: startedAt.toISOString(),
      dryRun: DRY_RUN,
      total: items.length,
      startedAt,
      items: []
    });
    let processed = 0;
    for (let i = 0; i < items.length; i++) {
      const { to, summary } = items[i] || {};
      if (!to || !summary) {
        console.warn(`${logPrefix} Skipping invalid item at index ${i}:`, items[i]);
        run.items.push({ to: String(to || ''), message: String(summary || ''), status: 'skipped', attempts: 0, attemptsLog: [{ attempt: 0, code: null, reason: 'Invalid item shape', timestamp: new Date() }] });
        continue;
      }
      const normalizedTo = normalizePhone(to);
      if (!normalizedTo) {
        console.warn(`${logPrefix} (${i + 1}/${items.length}) Skipping item with un-normalizable phone to=${to}.`);
        run.items.push({ to: String(to || ''), message: String(summary || ''), status: 'skipped', attempts: 0, attemptsLog: [{ attempt: 0, code: null, reason: 'Un-normalizable phone', timestamp: new Date() }] });
        continue;
      }
      if (!isValidEgyptPhone(normalizedTo)) {
        console.warn(`${logPrefix} (${i + 1}/${items.length}) Skipping invalid Egyptian phone to=${normalizedTo}.`);
        run.items.push({ to: normalizedTo, message: String(summary || ''), status: 'skipped', attempts: 0, attemptsLog: [{ attempt: 0, code: null, reason: 'Invalid Egyptian phone', timestamp: new Date() }] });
        continue;
      }
      console.log(`${logPrefix} (${i + 1}/${items.length}) Sending message to ${normalizedTo} ...`);
      try {
        const result = await sendWithRetries(normalizedTo, summary, 3);
        processed++;
        if (result.ok) {
          run.items.push({ to: normalizedTo, message: String(summary || ''), status: DRY_RUN ? 'dry_run' : 'sent', attempts: result.attempts, attemptsLog: result.attemptsLog, sentAt: new Date() });
          run.successCount = (run.successCount || 0) + 1;
        } else {
          const lastReason = result.attemptsLog?.slice(-1)[0]?.reason || 'Unknown failure';
          run.items.push({ to: normalizedTo, message: String(summary || ''), status: 'failed', attempts: result.attempts, attemptsLog: result.attemptsLog });
          run.failCount = (run.failCount || 0) + 1;
          console.error(`${logPrefix} Final failure to=${normalizedTo}: ${lastReason}`);
        }
      } catch (err) {
        // Already logged inside sendMessage; continue to next
        run.items.push({ to: normalizedTo, message: String(summary || ''), status: 'failed', attempts: 1, attemptsLog: [{ attempt: 1, code: err?.response?.status || null, reason: formatErrorReason(err), timestamp: new Date() }] });
        run.failCount = (run.failCount || 0) + 1;
      }
      if (i < items.length - 1 && SEND_DELAY_MS > 0) {
        console.log(`${logPrefix} Waiting ${SEND_DELAY_MS}ms before next message...`);
        await sleep(SEND_DELAY_MS);
      }
    }

    const endedAt = new Date();
    console.log(`${logPrefix} Job finished at ${endedAt.toISOString()} (processed ${processed}/${items.length})`);
    run.processed = processed;
    run.endedAt = endedAt;
    try {
      await run.save();
      console.log(`${logPrefix} Report saved: type=daily runId=${run.runId} success=${run.successCount || 0} failed=${run.failCount || 0}`);
    } catch (saveErr) {
      console.error(`${logPrefix} Failed to save report:`, saveErr.message);
    }
    return { processed };
  } catch (err) {
    console.error(`${logPrefix} Job failed:`, err.message);
    return { processed: 0, error: err.message };
  }
}

let scheduled = false;

function startDailyAutomation() {
  if (scheduled) {
    console.log(`${logPrefix} Scheduler already started; skipping.`);
    return;
  }
  if (!ENABLED) {
    console.log(`${logPrefix} Disabled via env DAILY_AUTOMATION_ENABLED=false`);
    return;
  }

  const options = {};
  if (CRON_TZ) {
    options.timezone = CRON_TZ;
    console.log(`${logPrefix} Using timezone: ${CRON_TZ}`);
  }

  cron.schedule(CRON_SCHEDULE, () => {
    runDailySummaryOnce();
  }, options);

  scheduled = true;
  console.log(`${logPrefix} Scheduler started (cron: "${CRON_SCHEDULE}") TZ=${CRON_TZ}.`);
}

// Auto-start on import if enabled
try {
  startDailyAutomation();
} catch (e) {
  console.error(`${logPrefix} Failed to start scheduler:`, e.message);
}

module.exports = {
  startDailyAutomation,
  runDailySummaryOnce
};