const User = require('../models/User');
const PricingPlan = require('../models/PricingPlan');
const ActivationRequest = require('../models/ActivationRequest');
const Discount = require('../models/Discount');
const Coupon = require('../models/Coupon');
const { sendEmail } = require('../utils/emailService');

/**
 * Get all pricing plans
 */
const getPricingPlans = async (req, res) => {
    try {
        const plans = await PricingPlan.find({ active: true }).sort({ monthlyPrice: 1 });
        res.status(200).json({
            success: true,
            plans
        });
    } catch (error) {
        console.error('Error fetching pricing plans:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching pricing plans',
            error: error.message
        });
    }
};

/**
 * Get user's current subscription status
 */
const getSubscriptionStatus = async (req, res) => {
    try {
        const user = await User.findById(req.user._id);
        
        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        const now = new Date();
        const isPremiumActive = user.premiumActive && 
                               user.premiumExpiry && 
                               new Date(user.premiumExpiry) > now;

        // Check trial status
        const isTrialActive = user.trialUsed && 
                             user.trialEndDate && 
                             new Date(user.trialEndDate) > now;
        
        // Calculate trial days remaining
        let trialDaysRemaining = 0;
        if (isTrialActive) {
            trialDaysRemaining = Math.ceil((new Date(user.trialEndDate) - now) / (1000 * 60 * 60 * 24));
        }

        // Auto-downgrade if expired
        if (user.premiumActive && user.premiumExpiry && new Date(user.premiumExpiry) <= now) {
            user.premiumActive = false;
            user.planType = 'free';
            await user.save();
        }

        // Calculate AI usage for free users (not premium and not on trial)
        let aiUsageInfo = null;
        if (!isPremiumActive && !isTrialActive) {
            // Reset daily counter if it's a new day
            const lastReset = user.aiUsage?.lastResetDate || new Date();
            const isNewDay = lastReset.toDateString() !== now.toDateString();
            
            if (isNewDay) {
                user.aiUsage = {
                    dailyQueries: 0,
                    lastResetDate: now
                };
                await user.save();
            }

            const dailyQueries = user.aiUsage?.dailyQueries || 0;
            const maxQueries = 10;
            aiUsageInfo = {
                dailyQueries: dailyQueries,
                maxQueries: maxQueries,
                remainingQueries: Math.max(0, maxQueries - dailyQueries)
            };
        }

        res.status(200).json({
            success: true,
            subscription: {
                premiumActive: isPremiumActive,
                planType: user.planType,
                premiumExpiry: user.premiumExpiry,
                daysRemaining: user.premiumExpiry && new Date(user.premiumExpiry) > now 
                    ? Math.ceil((new Date(user.premiumExpiry) - now) / (1000 * 60 * 60 * 24))
                    : 0,
                // Trial information
                trialActive: isTrialActive,
                trialStartDate: user.trialStartDate,
                trialEndDate: user.trialEndDate,
                trialDaysRemaining: trialDaysRemaining,
                trialUsed: user.trialUsed || false,
                aiUsage: aiUsageInfo
            }
        });
    } catch (error) {
        console.error('Error fetching subscription status:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching subscription status',
            error: error.message
        });
    }
};

/**
 * Calculate price with discounts and coupons
 */
const calculatePrice = async (req, res) => {
    try {
        const { planId, durationType, months, couponCode, employeeCount } = req.body;

        if (!planId || !durationType || !months) {
            return res.status(400).json({
                success: false,
                message: 'Missing required fields: planId, durationType, months'
            });
        }

        const plan = await PricingPlan.findOne({ planId, active: true });
        if (!plan) {
            return res.status(404).json({
                success: false,
                message: 'Pricing plan not found'
            });
        }

        // Calculate base price
        let basePrice = 0;
        if (durationType === 'monthly') {
            basePrice = plan.monthlyPrice * months;
        } else if (durationType === 'yearly') {
            basePrice = plan.yearlyPrice * Math.ceil(months / 12);
        } else {
            // Custom duration
            const monthlyRate = plan.monthlyPrice;
            basePrice = monthlyRate * months;
        }

        // Apply employee pricing for company plans
        if (plan.planId.includes('company') && employeeCount) {
            const employeePrice = plan.pricePerEmployee * employeeCount * months;
            basePrice = employeePrice;
        }

        // Apply active discount
        let discountAmount = 0;
        const discount = await Discount.findOne({
            type: durationType,
            active: true,
            $or: [
                { applicablePlans: { $in: [planId] } },
                { applicablePlans: { $size: 0 } }
            ],
            $or: [
                { endDate: { $exists: false } },
                { endDate: { $gte: new Date() } }
            ]
        }).sort({ createdAt: -1 });

        if (discount) {
            if (discount.discountType === 'percentage') {
                discountAmount = (basePrice * discount.value) / 100;
            } else {
                discountAmount = discount.value;
            }
        }

        // Apply coupon if provided
        let couponDiscount = 0;
        if (couponCode) {
            const coupon = await Coupon.findOne({
                code: couponCode.toUpperCase().trim(),
                active: true,
                expiryDate: { $gte: new Date() }
            });

            if (coupon) {
                // Check if user already used this coupon
                const user = await User.findById(req.user._id);
                const alreadyUsed = coupon.usedBy.some(u => u.user && u.user.toString() === user._id.toString());
                
                if (alreadyUsed) {
                    return res.status(400).json({
                        success: false,
                        message: 'Coupon already used'
                    });
                }

                // Check if coupon applies to this plan
                if (coupon.applicablePlans.length > 0 && !coupon.applicablePlans.includes(planId)) {
                    return res.status(400).json({
                        success: false,
                        message: 'Coupon not applicable to selected plan'
                    });
                }

                // Check max uses
                if (coupon.usedBy.length >= coupon.maxUses) {
                    return res.status(400).json({
                        success: false,
                        message: 'Coupon has reached maximum uses'
                    });
                }

                if (coupon.discountType === 'percentage') {
                    couponDiscount = ((basePrice - discountAmount) * coupon.value) / 100;
                } else {
                    couponDiscount = coupon.value;
                }
            } else {
                return res.status(404).json({
                    success: false,
                    message: 'Invalid or expired coupon code'
                });
            }
        }

        const finalPrice = Math.max(0, basePrice - discountAmount - couponDiscount);

        res.status(200).json({
            success: true,
            calculation: {
                basePrice,
                discountAmount,
                couponDiscount,
                finalPrice,
                currency: 'EGP'
            }
        });
    } catch (error) {
        console.error('Error calculating price:', error);
        res.status(500).json({
            success: false,
            message: 'Error calculating price',
            error: error.message
        });
    }
};

/**
 * Create activation request
 */
const createActivationRequest = async (req, res) => {
    try {
        const { planId, durationType, months, paymentScreenshotURL, couponCode, employeeCount } = req.body;

        if (!planId || !durationType || !months || !paymentScreenshotURL) {
            return res.status(400).json({
                success: false,
                message: 'Missing required fields'
            });
        }

        const user = await User.findById(req.user._id);
        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        // Calculate price
        const plan = await PricingPlan.findOne({ planId, active: true });
        if (!plan) {
            return res.status(404).json({
                success: false,
                message: 'Pricing plan not found'
            });
        }

        let basePrice = 0;
        if (durationType === 'monthly') {
            basePrice = plan.monthlyPrice * months;
        } else if (durationType === 'yearly') {
            basePrice = plan.yearlyPrice * Math.ceil(months / 12);
        } else {
            basePrice = plan.monthlyPrice * months;
        }

        if (plan.planId.includes('company') && employeeCount) {
            basePrice = plan.pricePerEmployee * employeeCount * months;
        }

        // Apply discounts and coupons
        let discountAmount = 0;
        let couponDiscount = 0;
        let finalCoupon = null;

        const discount = await Discount.findOne({
            type: durationType,
            active: true,
            $or: [
                { applicablePlans: { $in: [planId] } },
                { applicablePlans: { $size: 0 } }
            ]
        }).sort({ createdAt: -1 });

        if (discount) {
            if (discount.discountType === 'percentage') {
                discountAmount = (basePrice * discount.value) / 100;
            } else {
                discountAmount = discount.value;
            }
        }

        if (couponCode) {
            const coupon = await Coupon.findOne({
                code: couponCode.toUpperCase().trim(),
                active: true,
                expiryDate: { $gte: new Date() }
            });

            if (coupon) {
                const alreadyUsed = coupon.usedBy.some(u => u.user && u.user.toString() === user._id.toString());
                if (!alreadyUsed && 
                    (coupon.applicablePlans.length === 0 || coupon.applicablePlans.includes(planId)) &&
                    coupon.usedBy.length < coupon.maxUses) {
                    finalCoupon = coupon;
                    if (coupon.discountType === 'percentage') {
                        couponDiscount = ((basePrice - discountAmount) * coupon.value) / 100;
                    } else {
                        couponDiscount = coupon.value;
                    }
                }
            }
        }

        const finalPrice = Math.max(0, basePrice - discountAmount - couponDiscount);

        // Create activation request
        const activationRequest = new ActivationRequest({
            user: user._id,
            userEmail: user.email,
            planId,
            durationType,
            months,
            paymentScreenshotURL,
            requestedAmount: basePrice,
            discountApplied: discountAmount,
            couponCode: finalCoupon ? finalCoupon.code : null,
            finalAmount: finalPrice
        });

        await activationRequest.save();

        res.status(201).json({
            success: true,
            message: 'Activation request created successfully',
            activationRequest
        });
    } catch (error) {
        console.error('Error creating activation request:', error);
        res.status(500).json({
            success: false,
            message: 'Error creating activation request',
            error: error.message
        });
    }
};

/**
 * Get user's activation requests
 */
const getUserActivationRequests = async (req, res) => {
    try {
        const requests = await ActivationRequest.find({ user: req.user._id })
            .sort({ createdAt: -1 })
            .populate('approvedBy', 'name email');

        res.status(200).json({
            success: true,
            requests
        });
    } catch (error) {
        console.error('Error fetching activation requests:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching activation requests',
            error: error.message
        });
    }
};

/**
 * Admin: Get all activation requests
 */
const getAllActivationRequests = async (req, res) => {
    try {
        const { status, page = 1, limit = 20 } = req.query;
        const query = status && status !== 'all' ? { status } : {};

        const requests = await ActivationRequest.find(query)
            .sort({ createdAt: -1 })
            .limit(limit * 1)
            .skip((page - 1) * limit)
            .populate('user', 'name email')
            .populate('approvedBy', 'name email');

        const total = await ActivationRequest.countDocuments(query);

        res.status(200).json({
            success: true,
            requests,
            totalPages: Math.ceil(total / limit),
            currentPage: page,
            total
        });
    } catch (error) {
        console.error('Error fetching activation requests:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching activation requests',
            error: error.message
        });
    }
};

/**
 * Admin: Approve activation request
 */
const approveActivationRequest = async (req, res) => {
    try {
        const { requestId } = req.params;
        const { months } = req.body; // Allow admin to override duration

        const activationRequest = await ActivationRequest.findById(requestId)
            .populate('user');

        if (!activationRequest) {
            return res.status(404).json({
                success: false,
                message: 'Activation request not found'
            });
        }

        if (activationRequest.status !== 'pending') {
            return res.status(400).json({
                success: false,
                message: 'Activation request is not pending'
            });
        }

        const user = activationRequest.user;
        const durationMonths = months || activationRequest.months;

        // Calculate expiry date
        const expiryDate = new Date();
        expiryDate.setMonth(expiryDate.getMonth() + durationMonths);

        // Update user premium status
        user.premiumActive = true;
        user.premiumExpiry = expiryDate;
        user.planType = activationRequest.planId;

        // Add to premium history
        user.premiumHistory.push({
            planType: activationRequest.planId,
            activatedAt: new Date(),
            expiredAt: expiryDate,
            durationMonths
        });

        await user.save();

        // Update activation request
        activationRequest.status = 'approved';
        activationRequest.activatedAt = new Date();
        activationRequest.approvedBy = req.user._id;
        await activationRequest.save();

        // Mark coupon as used if applicable
        if (activationRequest.couponCode) {
            const coupon = await Coupon.findOne({ code: activationRequest.couponCode });
            if (coupon) {
                coupon.usedBy.push({
                    user: user._id,
                    usedAt: new Date()
                });
                await coupon.save();
            }
        }

        // Send activation email
        try {
            await sendActivationEmail(user, activationRequest, expiryDate);
        } catch (emailError) {
            console.error('Error sending activation email:', emailError);
        }

        res.status(200).json({
            success: true,
            message: 'Activation request approved successfully',
            activationRequest
        });
    } catch (error) {
        console.error('Error approving activation request:', error);
        res.status(500).json({
            success: false,
            message: 'Error approving activation request',
            error: error.message
        });
    }
};

/**
 * Admin: Reject activation request
 */
const rejectActivationRequest = async (req, res) => {
    try {
        const { requestId } = req.params;
        const { rejectionReason } = req.body;

        const activationRequest = await ActivationRequest.findById(requestId);

        if (!activationRequest) {
            return res.status(404).json({
                success: false,
                message: 'Activation request not found'
            });
        }

        if (activationRequest.status !== 'pending') {
            return res.status(400).json({
                success: false,
                message: 'Activation request is not pending'
            });
        }

        activationRequest.status = 'rejected';
        activationRequest.rejectedAt = new Date();
        activationRequest.rejectionReason = rejectionReason || 'Payment verification failed';

        await activationRequest.save();

        res.status(200).json({
            success: true,
            message: 'Activation request rejected',
            activationRequest
        });
    } catch (error) {
        console.error('Error rejecting activation request:', error);
        res.status(500).json({
            success: false,
            message: 'Error rejecting activation request',
            error: error.message
        });
    }
};

/**
 * Helper: Send activation email
 */
const sendActivationEmail = async (user, activationRequest, expiryDate) => {
    try {
        const plan = await PricingPlan.findOne({ planId: activationRequest.planId });
        const subject = 'TaskTree Premium Activation Confirmed';
        const html = `
            <h2>Premium Activation Confirmed!</h2>
            <p>Hello ${user.name},</p>
            <p>Your premium subscription has been activated successfully!</p>
            <h3>Subscription Details:</h3>
            <ul>
                <li><strong>Plan:</strong> ${plan?.name || activationRequest.planId}</li>
                <li><strong>Duration:</strong> ${activationRequest.months} months</li>
                <li><strong>Expiry Date:</strong> ${expiryDate.toLocaleDateString()}</li>
                <li><strong>Amount Paid:</strong> ${activationRequest.finalAmount} EGP</li>
            </ul>
            <p>Thank you for choosing TaskTree Premium!</p>
        `;

        await sendEmail(user.email, subject, html);
    } catch (error) {
        console.error('Error sending activation email:', error);
        throw error;
    }
};

module.exports = {
    getPricingPlans,
    getSubscriptionStatus,
    calculatePrice,
    createActivationRequest,
    getUserActivationRequests,
    getAllActivationRequests,
    approveActivationRequest,
    rejectActivationRequest
};

