const User = require('../models/User');
const GroupTask = require('../models/GroupTask');
const moment = require('moment');

// Robust import for phone normalizer
const phoneUtils = require('../utils/phoneUtils');
const normalizeEgyptPhone =
  typeof phoneUtils === 'function'
    ? phoneUtils
    : phoneUtils && typeof phoneUtils.normalizeEgyptPhone === 'function'
      ? phoneUtils.normalizeEgyptPhone
      : null;

// GET /api/morning-message - Generate morning message for all users
const getMorningMessage = async (req, res) => {
  try {
    // Get active users with phone numbers
    const users = await User.find({
      phone: { $exists: true, $ne: null, $ne: '' },
      status: 'active'
    }).select('_id name phone email');

    if (!users || users.length === 0) {
      return res.status(200).json([]);
    }

    const messages = [];

    for (const user of users) {
      try {
        const msg = await generateUserMorningMessage(user);
        if (msg) messages.push(msg);
      } catch (err) {
        console.error(`Error generating morning message for user ${user._id}:`, err);
      }
    }

    res.status(200).json(messages);
  } catch (error) {
    console.error('Error generating morning messages:', error);
    res.status(500).json({ error: 'Internal server error', message: 'Failed to generate morning messages' });
  }
};

// Generate morning message for a single user
const generateUserMorningMessage = async (user) => {
  // Dates for today
  const today = moment().startOf('day');
  const endOfToday = moment().endOf('day');

  // Personal tasks due today
  const Section = require('../models/Sections');
  const userSections = await Section.find({ userId: user._id });

  let todayPersonalTasks = [];
  userSections.forEach(section => {
    section.tasks.forEach(task => {
      if (task.dueDate) {
        const due = moment(task.dueDate);
        if (due.isSameOrAfter(today) && due.isSameOrBefore(endOfToday) && !task.isDone) {
          todayPersonalTasks.push({ ...task.toObject(), sectionName: section.name });
        }
      }
    });
  });

  // Group tasks due today
  const todayGroupTasks = await GroupTask.find({
    assignedTo: user.email,
    dueDate: { $gte: today.toDate(), $lte: endOfToday.toDate() },
    completed: false
  }).populate('group', 'name');

  const message = buildMorningMessage(user.name, [...todayPersonalTasks, ...todayGroupTasks]);

  // Normalize phone and skip invalid
  if (!normalizeEgyptPhone) {
    console.error(`☎️ Phone normalizer not available; skipping user ${user.name} (${user._id}).`);
    return null;
  }
  const normalizedPhone = normalizeEgyptPhone(user.phone);
  if (!normalizedPhone) {
    console.warn(`☎️ Skipping user ${user.name} (${user._id}) due to invalid phone: '${user.phone}'`);
    return null;
  }

  if (normalizedPhone !== user.phone) {
    console.log(`☎️ Normalized phone for ${user.name}: '${user.phone}' -> '${normalizedPhone}'`);
  }

  return { to: normalizedPhone, summary: message };
};

// Build professional, encouraging Arabic morning message
const buildMorningMessage = (name, todayTasks) => {
  let msg = `صباح الخير ${name} 🌅\n`;
  msg += '\n';

  if (todayTasks && todayTasks.length > 0) {
    msg += 'دي المهام المستحقة عليك النهاردة — خلّيك مركز وابدأ بالأهم:\n';
    todayTasks.forEach(task => {
      const title = task.title || task.name || 'مهمة بدون عنوان';
      msg += `- ${title}`;
      if (task.description) {
        msg += ` (${task.description})`;
      }
      if (task.sectionName) {
        msg += ` — قسم: ${task.sectionName}`;
      }
      if (task.group && task.group.name) {
        msg += ` — جروب: ${task.group.name}`;
      }
      msg += '\n';
    });
    msg += '\n';
    msg += 'نصيحة اليوم: حدّد وقت تركيز بدون مقاطعات، وابدأ بالمهمة ذات التأثير الأكبر. بالتوفيق! 💪\n';
  } else {
    msg += 'مفيش مهام مستحقة عليك النهاردة. استغل الوقت في التخطيط للأسبوع أو تطوير مهارة جديدة. يوم موفق! ✨\n';
  }

  return msg.trim();
};

module.exports = { getMorningMessage };