const Group = require('../models/Group');
const GroupMembership = require('../models/GroupMembership');
const GroupTask = require('../models/GroupTask');
const GroupSubtask = require('../models/GroupSubtask');
const GroupSection = require('../models/GroupSection');
const User = require('../models/User');
const ActivityLog = require('../models/ActivityLog');
const emailService = require('../utils/emailService');

// Create a new group and invite members
exports.createGroup = async (req, res) => {
    try {
        const { name, description, emails, canMembersViewAllTasks } = req.body;
        if (!name || !emails || !Array.isArray(emails) || emails.length === 0) {
            return res.status(400).json({ error: 'Group name and at least one member email are required.' });
        }
        if (!req.user || !req.user._id) {
            return res.status(401).json({ error: 'User authentication required.' });
        }

        // Check premium status for group limits
        const user = await User.findById(req.user._id);
        if (!user) {
            return res.status(404).json({ error: 'User not found' });
        }

        const now = new Date();
        const isPremiumActive = user.premiumActive && 
                               user.premiumExpiry && 
                               new Date(user.premiumExpiry) > now;

        // Free plan: Limited to 2 groups max
        if (!isPremiumActive) {
            const groupCount = await Group.countDocuments({ creator: req.user._id });
            if (groupCount >= 2) {
                return res.status(403).json({ 
                    error: 'Free plan limit reached. You can only create up to 2 groups. Upgrade to Premium for unlimited groups.',
                    code: 'GROUP_LIMIT_REACHED',
                    maxGroups: 2,
                    currentCount: groupCount
                });
            }
        }

        // Check member limit for free users (3 members per group including creator)
        if (!isPremiumActive) {
            const totalMembers = emails.length + 1; // +1 for creator
            if (totalMembers > 3) {
                return res.status(403).json({ 
                    error: 'Free plan limit: Each group can have a maximum of 3 members (including you). Upgrade to Premium for unlimited team collaboration.',
                    code: 'GROUP_MEMBER_LIMIT_REACHED',
                    maxMembers: 3,
                    requestedCount: totalMembers
                });
            }
        }

        // Debug log
        console.log('Creating group:', { name, description, emails, user: req.user });
        const creator = req.user._id;
        const group = await Group.create({
            name,
            description,
            creator,
            members: emails,
            canMembersViewAllTasks: !!canMembersViewAllTasks
        });
        // Add memberships
        await GroupMembership.create({
            group: group._id,
            user: creator,
            email: req.user.email,
            role: 'creator',
            canViewAllTasks: !!canMembersViewAllTasks
        });
        for (const email of emails) {
            if (email !== req.user.email) {
                // Check if user exists in the system
                const existingUser = await User.findOne({ email: email.toLowerCase().trim() });
                
                const membershipData = {
                    group: group._id,
                    email: email.toLowerCase().trim(),
                    role: 'member',
                    canViewAllTasks: !!canMembersViewAllTasks
                };
                
                if (existingUser) {
                    membershipData.user = existingUser._id;
                }
                
                await GroupMembership.create(membershipData);
                // Send invitation email (do not block group creation if email fails)
                try {
                    await emailService.sendMail({
                        to: email,
                        subject: `Added to group: ${name}`,
                        text: `You have been added to the group "${name}"
Description: ${description}
Created by: ${req.user.name || req.user.email}`
                    });
                } catch (emailErr) {
                    console.error('Failed to send group invite email:', emailErr);
                }
            }
        }
        res.status(201).json(group);
    } catch (err) {
        console.error('Error creating group:', err);
        res.status(500).json({ error: err.message });
    }
};

// List groups for current user
exports.getGroups = async (req, res) => {
    try {
        // Find groups where user is a member by email or user ID
        const memberships = await GroupMembership.find({
            $or: [
                { email: req.user.email },
                { user: req.user._id }
            ]
        }).populate({
            path: 'group',
            populate: {
                path: 'creator',
                select: 'name email'
            }
        });
        
        const groups = memberships.map(m => m.group).filter(group => group !== null);
        res.json(groups);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Check if user emails exist
exports.checkUserEmails = async (req, res) => {
    try {
        const { emails } = req.body;
        if (!Array.isArray(emails)) return res.status(400).json({ error: 'Emails must be an array' });
        const users = await User.find({ email: { $in: emails } }, 'email');
        const found = users.map(u => u.email);
        const notFound = emails.filter(e => !found.includes(e));
        res.json({ found, notFound });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Get full group details (group, members, tasks, subtasks, analysis)
exports.getGroupDetails = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const group = await Group.findById(groupId).populate('creator', 'name email');
        if (!group) return res.status(404).json({ error: 'Group not found' });

        // Get members with their roles (populate user info if possible)
        const memberships = await GroupMembership.find({ group: groupId })
            .populate('user', 'name email');
        
        const members = [];
        const admins = [];
        
        memberships.forEach(m => {
            const memberData = m.user ? { 
                _id: m.user._id, 
                name: m.user.name, 
                email: m.user.email,
                role: m.role 
            } : { 
                email: m.email,
                role: m.role 
            };
            
            members.push(memberData);
            
            // Add to admins if role is creator (since there's no separate admin role)
            if (m.role === 'creator') {
                admins.push(memberData);
            }
        });

        // Get tasks (populate assignedTo)
        const tasks = await GroupTask.find({ group: groupId })
            .populate('assignedTo', 'name email');

        // Get subtasks (filter by groupTask for each task)
        const subtasks = await GroupSubtask.find({ groupTask: { $in: tasks.map(t => t._id) } });

        // Analysis
        const totalTasks = tasks.length;
        const completedTasks = tasks.filter(t => t.done).length;
        const pendingTasks = totalTasks - completedTasks;
        const analysis = { totalTasks, completedTasks, pendingTasks };

        // Add admins array to group object
        const groupWithAdmins = {
            ...group.toObject(),
            admins
        };

        res.json({ group: groupWithAdmins, members, tasks, subtasks, analysis });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Delete a group and all related data
exports.deleteGroup = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found' });
        // Only creator can delete
        if (group.creator.toString() !== req.user._id.toString()) {
            return res.status(403).json({ error: 'Only the group creator can delete this group.' });
        }
        // Delete all group memberships
        await GroupMembership.deleteMany({ group: groupId });
        // Delete all group tasks
        await GroupTask.deleteMany({ group: groupId });
        // Delete all group subtasks
        await GroupSubtask.deleteMany({ group: groupId });
        // Delete the group itself
        await Group.findByIdAndDelete(groupId);
        res.json({ message: 'Group and all related data deleted.' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Create a new group task
exports.createGroupTask = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const { title, description, assignedTo, priority, dueDate, points, sectionId } = req.body;
        if (!title) return res.status(400).json({ error: 'Task title is required.' });
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found.' });
        
        // Check if user is admin (creator or admin role)
        const userEmail = (req.user.email || '').trim().toLowerCase();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!membership || (membership.role !== 'creator' && membership.role !== 'admin')) {
            return res.status(403).json({ error: 'Only group admins can create tasks.' });
        }
        
        // Validate sectionId if provided
        if (sectionId) {
            const section = await GroupSection.findOne({ _id: sectionId, groupId: groupId });
            if (!section) {
                return res.status(400).json({ error: 'Invalid section ID.' });
            }
        }
        
        // Handle assignedTo array
        let assignedEmails = [];
        if (assignedTo && Array.isArray(assignedTo)) {
            for (const member of assignedTo) {
                if (typeof member === 'string') {
                    assignedEmails.push(member.toLowerCase());
                } else if (member._id) {
                    // If it's a user object, get the email
                    const user = await User.findById(member._id || member);
                    if (user && user.email) assignedEmails.push(user.email.toLowerCase());
                }
            }
        }
        
        const task = await GroupTask.create({
            group: groupId,
            sectionId: sectionId || null,
            title,
            description,
            assignedTo: assignedEmails,
            createdBy: req.user._id,
            priority: priority || 'medium',
            status: 'pending',
            completed: false,
            dueDate: dueDate ? new Date(dueDate) : null,
            points: parseInt(points) || 0,
        });
        
        // Log activity
        try {
            await ActivityLog.logActivity({
                userId: req.user._id,
                action: 'group_task_created',
                taskTitle: title,
                taskDescription: description,
                groupId: groupId,
                groupName: group.name,
                metadata: { priority, points: parseInt(points) || 0 }
            });
        } catch (logErr) {
            console.error('Error logging activity:', logErr);
        }
        
        // Send email notifications to assigned members
        for (const email of assignedEmails) {
            try {
                await emailService.sendMail({
                    to: email,
                    subject: `New task assigned: ${title}`,
                    text: `You have been assigned a new task in group "${group.name}":

Task: ${title}
Description: ${description || 'No description'}
Due Date: ${dueDate ? new Date(dueDate).toLocaleDateString() : 'No due date'}
Points: ${points || 0}`
                });
            } catch (emailErr) {
                console.error('Failed to send task assignment email:', emailErr);
            }
        }
        
        res.status(201).json(task);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Get all tasks for a group
exports.getGroupTasks = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found.' });
        
        // Check if user is a member
        const userEmail = (req.user.email || '').trim().toLowerCase();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!membership) {
            return res.status(403).json({ error: 'Only group members can view tasks.' });
        }
        
        const tasks = await GroupTask.find({ group: groupId })
            .populate('createdBy', 'name email')
            .sort({ createdAt: -1 });
        
        // Filter tasks based on visibility permissions
        let filteredTasks = tasks;
        if (!group.canMembersViewAllTasks && membership.role !== 'creator' && membership.role !== 'admin') {
            filteredTasks = tasks.filter(task => 
                task.assignedTo.includes(userEmail) ||
                task.createdBy._id.toString() === req.user._id.toString()
            );
        }
        
        res.json(filteredTasks);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Add a subtask to a group task
exports.addGroupSubtask = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const taskId = req.params.taskId;
        const { title, done } = req.body;
        if (!title) return res.status(400).json({ error: 'Subtask title is required.' });
        // Check if user is a group member
        const userEmail = req.user.email.trim().toLowerCase();
        const isMember = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!isMember) {
            return res.status(403).json({ error: 'Only group members can add subtasks.' });
        }
        // Create the subtask (fix: use correct field names)
        const subtask = await GroupSubtask.create({
            groupTask: taskId,
            title,
            status: done ? 'completed' : 'pending',
        });
        // Add subtask to the parent task's subtasks array
        await GroupTask.findByIdAndUpdate(taskId, { $push: { subtasks: subtask._id } });
        res.status(201).json(subtask);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Update task status (completed/not completed)
exports.updateTaskStatus = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const taskId = req.params.taskId;
        const { completed } = req.body;
        const userId = req.user._id;
        const userEmail = req.user.email.toLowerCase();
        
        const task = await GroupTask.findById(taskId);
        if (!task || task.group.toString() !== groupId) {
            return res.status(404).json({ error: 'Task not found.' });
        }
        
        // Check if user can update this task status
        const isAssigned = (task.assignedTo || []).includes(userEmail);
        const isCreator = task.createdBy.toString() === userId.toString();
        
        // Check if user is admin
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        const isAdmin = membership && (membership.role === 'creator' || membership.role === 'admin');
        
        if (!isAssigned && !isCreator && !isAdmin) {
            return res.status(403).json({ error: 'Only assigned users or admins can update task status.' });
        }
        
        task.completed = completed;
        task.status = completed ? 'completed' : 'pending';
        await task.save();

        // Log activity when group task is completed
        if (completed) {
            try {
                await ActivityLog.logActivity({
                    userId: req.user._id,
                    action: 'group_task_completed',
                    taskTitle: task.title,
                    taskDescription: task.description,
                    taskId: task._id,
                    groupId: groupId,
                    metadata: {
                        taskType: 'group_task'
                    }
                });
                console.log(`Activity logged for completed group task: ${task.title}`);
            } catch (logError) {
                console.error('Error logging group task activity:', logError);
                // Don't fail the main operation if logging fails
            }
        }
        
        res.json({ message: `Task marked as ${completed ? 'completed' : 'pending'}.`, task });
    } catch (err) {
        console.error('Error in updateTaskStatus:', err);
        res.status(500).json({ error: err.message });
    }
};

// Mark a group task as done/undone (legacy endpoint)
exports.markGroupTaskDone = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const taskId = req.params.taskId;
        const userId = req.user._id;
        const task = await GroupTask.findById(taskId);
        if (!task) return res.status(404).json({ error: 'Task not found.' });
        // Only assigned user or creator can mark as done
        const isAssigned = (task.assignedTo || []).includes(req.user.email);
        const isOwner = task.createdBy.toString() === userId.toString();
        if (!isAssigned && !isOwner) {
            return res.status(403).json({ error: 'Only assigned user or creator can mark as done.' });
        }
        // Toggle done
        const newCompleted = !task.completed;
        task.completed = newCompleted;
        task.status = newCompleted ? 'completed' : 'pending';
        await task.save();

        // Log activity when group task is completed
        if (newCompleted) {
            try {
                await ActivityLog.logActivity({
                    userId: req.user._id,
                    action: 'task_completed',
                    taskTitle: task.title,
                    taskDescription: task.description,
                    taskId: task._id,
                    groupId: groupId,
                    metadata: {
                        taskType: 'group_task'
                    }
                });
                console.log(`Activity logged for completed group task: ${task.title}`);
            } catch (logError) {
                console.error('Error logging group task activity:', logError);
                // Don't fail the main operation if logging fails
            }
        }

        res.json({ message: `Task marked as ${task.status}.`, task });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Mark a group subtask as done/undone
exports.markGroupSubtaskDone = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const taskId = req.params.taskId;
        const subtaskId = req.params.subtaskId;
        const userId = req.user._id;
        const subtask = await GroupSubtask.findById(subtaskId);
        if (!subtask) return res.status(404).json({ error: 'Subtask not found.' });
        const task = await GroupTask.findById(taskId);
        if (!task) return res.status(404).json({ error: 'Task not found.' });
        // Only assigned user or task owner can mark as done
        const isAssigned = (subtask.assignedTo || []).includes(req.user.email);
        const isOwner = task.createdBy.toString() === userId.toString();
        if (!isAssigned && !isOwner) {
            return res.status(403).json({ error: 'Only assigned user or task owner can mark as done.' });
        }
        // Toggle done
        const newStatus = subtask.status === 'completed' ? 'pending' : 'completed';
        subtask.status = newStatus;
        await subtask.save();
        res.json({ message: `Subtask marked as ${newStatus}.`, subtask });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Update group details
exports.updateGroup = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const { name, description, emails } = req.body;
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found' });
        // Only creator can update
        if (group.creator.toString() !== req.user._id.toString()) {
            return res.status(403).json({ error: 'Only the group creator can update this group.' });
        }
        if (name) group.name = name;
        if (description) group.description = description;
        // Optionally update members if emails provided (not removing existing, just adding new)
        if (Array.isArray(emails)) {
            for (const email of emails) {
                if (!group.members.includes(email)) {
                    group.members.push(email);
                    await GroupMembership.create({
                        group: group._id,
                        email,
                        role: 'member',
                        canViewAllTasks: group.canMembersViewAllTasks
                    });
                }
            }
        }
        await group.save();
        res.json(group);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Remove a member from a group
exports.removeGroupMember = async (req, res) => {
    try {
        const groupId = req.params.groupId;
        const memberId = req.params.memberId;
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found' });
        // Only creator can remove members
        if (group.creator.toString() !== req.user._id.toString()) {
            return res.status(403).json({ error: 'Only the group creator can remove members.' });
        }
        // Remove from GroupMembership by _id or email
        let membership;
        if (memberId.match(/^[0-9a-fA-F]{24}$/)) {
            membership = await GroupMembership.findOneAndDelete({ group: groupId, user: memberId });
        } else {
            membership = await GroupMembership.findOneAndDelete({ group: groupId, email: memberId });
        }
        if (!membership) return res.status(404).json({ error: 'Member not found in group.' });
        // Remove from group.members array if present
        group.members = group.members.filter(m => m.toString() !== memberId);
        await group.save();
        res.json({ message: 'Member removed from group.' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// ... More controllers for group tasks, subtasks, permissions, etc. ...

exports.updateGroupTask = async (req, res) => {
    try {
        const { groupId, taskId } = req.params;
        const { title, description, priority, dueDate, assignedTo, points, sectionId } = req.body;
        const userId = req.user._id;
        const userEmail = req.user.email.toLowerCase();
        
        const task = await GroupTask.findById(taskId);
        if (!task || task.group.toString() !== groupId) {
            return res.status(404).json({ error: 'Task not found.' });
        }
        
        // Check if user can update this task (creator or admin)
        const isCreator = task.createdBy.toString() === userId.toString();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        const isAdmin = membership && (membership.role === 'creator' || membership.role === 'admin');
        
        if (!isCreator && !isAdmin) {
            return res.status(403).json({ error: 'Only task creator or group admin can update task.' });
        }
        
        // Validate sectionId if provided
        if (sectionId !== undefined) {
            if (sectionId === null || sectionId === '') {
                task.sectionId = null;
            } else {
                const section = await GroupSection.findOne({ _id: sectionId, groupId: groupId });
                if (!section) {
                    return res.status(400).json({ error: 'Invalid section ID.' });
                }
                task.sectionId = sectionId;
            }
        }
        
        // Update fields
        if (title !== undefined) task.title = title;
        if (description !== undefined) task.description = description;
        if (priority !== undefined) task.priority = priority;
        if (dueDate !== undefined) task.dueDate = dueDate ? new Date(dueDate) : null;
        if (points !== undefined) task.points = parseInt(points) || 0;
        
        if (assignedTo !== undefined) {
            if (Array.isArray(assignedTo)) {
                // Handle array of emails or user objects
                let normalizedEmails = [];
                for (const member of assignedTo) {
                    if (typeof member === 'string') {
                        normalizedEmails.push(member.toLowerCase());
                    } else if (member._id) {
                        const user = await User.findById(member._id);
                        if (user?.email) normalizedEmails.push(user.email.toLowerCase());
                    }
                }
                
                // Validate all emails exist in group
                const memberships = await GroupMembership.find({ 
                    group: groupId, 
                    email: { $in: normalizedEmails } 
                });
                
                if (memberships.length !== normalizedEmails.length) {
                    return res.status(400).json({ error: 'One or more assigned users are not members of this group.' });
                }
                
                task.assignedTo = normalizedEmails;
            } else if (assignedTo) {
                // Single assignment - handle userId or email
                let emails = [];
                try {
                    const user = await User.findById(assignedTo);
                    if (user?.email) emails = [user.email.toLowerCase()];
                } catch (_) {
                    if (typeof assignedTo === 'string') emails = [assignedTo.toLowerCase()];
                }
                
                // Validate email exists in group
                if (emails.length > 0) {
                    const membership = await GroupMembership.findOne({ group: groupId, email: emails[0] });
                    if (!membership) {
                        return res.status(400).json({ error: 'Assigned user is not a member of this group.' });
                    }
                }
                
                task.assignedTo = emails;
            } else {
                task.assignedTo = [];
            }
        }
        
        await task.save();
        res.json({ message: 'Task updated successfully.', task });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

exports.deleteGroupTask = async (req, res) => {
    try {
        const { groupId, taskId } = req.params;
        const task = await GroupTask.findById(taskId);
        if (!task || task.group.toString() !== groupId) return res.status(404).json({ error: 'Task not found.' });
        // Only creator can delete
        if (task.createdBy.toString() !== req.user._id.toString()) return res.status(403).json({ error: 'Only the task creator can delete this task.' });
        // Log activity before deletion
        try {
            const group = await Group.findById(groupId);
            await ActivityLog.logActivity({
                userId: req.user._id,
                action: 'group_task_deleted',
                taskTitle: task.title,
                groupId: groupId,
                groupName: group?.name,
                metadata: {}
            });
        } catch (logErr) {
            console.error('Error logging activity:', logErr);
        }
        
        // Delete subtasks
        await GroupSubtask.deleteMany({ groupTask: taskId });
        await GroupTask.findByIdAndDelete(taskId);
        res.json({ message: 'Task deleted.' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

exports.updateGroupSubtask = async (req, res) => {
    try {
        const { groupId, taskId, subtaskId } = req.params;
        const updates = req.body;
        const task = await GroupTask.findById(taskId);
        if (!task || task.group.toString() !== groupId) return res.status(404).json({ error: 'Task not found.' });
        // Only task creator can update subtask
        if (task.createdBy.toString() !== req.user._id.toString()) return res.status(403).json({ error: 'Only the task creator can update subtasks.' });
        const subtask = await GroupSubtask.findById(subtaskId);
        if (!subtask || subtask.groupTask.toString() !== taskId) return res.status(404).json({ error: 'Subtask not found.' });
        if (updates.title !== undefined) subtask.title = updates.title;
        if (updates.status !== undefined) subtask.status = updates.status;
        if (updates.assignedTo !== undefined) {
            // assignedTo is list of emails; allow single string
            if (Array.isArray(updates.assignedTo)) subtask.assignedTo = updates.assignedTo.map(e => e.toLowerCase());
            else if (typeof updates.assignedTo === 'string') subtask.assignedTo = [updates.assignedTo.toLowerCase()];
        }
        await subtask.save();
        res.json(subtask);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

exports.deleteGroupSubtask = async (req, res) => {
    try {
        const { groupId, taskId, subtaskId } = req.params;
        const task = await GroupTask.findById(taskId);
        if (!task || task.group.toString() !== groupId) return res.status(404).json({ error: 'Task not found.' });
        if (task.createdBy.toString() !== req.user._id.toString()) return res.status(403).json({ error: 'Only the task creator can delete subtasks.' });
        await GroupSubtask.findByIdAndDelete(subtaskId);
        await GroupTask.findByIdAndUpdate(taskId, { $pull: { subtasks: subtaskId } });
        res.json({ message: 'Subtask deleted.' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

exports.addGroupMember = async (req, res) => {
    try {
        const { groupId } = req.params;
        let { email } = req.body;
        
        // Handle case where email might be an object or string
        if (typeof email === 'object' && email !== null) {
            email = email.email || email.value || '';
        }
        if (typeof email !== 'string') {
            email = String(email || '');
        }
        
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found' });
        if (group.creator.toString() !== req.user._id.toString()) return res.status(403).json({ error: 'Only the group creator can add members.' });
        const normalizedEmail = (email || '').trim().toLowerCase();
        if (!normalizedEmail) return res.status(400).json({ error: 'Email is required.' });
        
        // Check premium status for team member limits
        const user = await User.findById(req.user._id);
        if (!user) return res.status(404).json({ error: 'User not found' });
        
        const now = new Date();
        const isPremiumActive = user.premiumActive && 
                               user.premiumExpiry && 
                               new Date(user.premiumExpiry) > now;
        
        // Free plan: Limited team collaboration (max 3 members per group)
        if (!isPremiumActive) {
            const currentMemberCount = group.members.length;
            if (currentMemberCount >= 3) {
                return res.status(403).json({ 
                    error: 'Free plan limit reached. You can only have up to 3 members per group. Upgrade to Premium for unlimited team collaboration.',
                    code: 'TEAM_MEMBER_LIMIT_REACHED',
                    maxMembers: 3,
                    currentCount: currentMemberCount
                });
            }
        }
        
        // Check if user exists in the system
        const existingUser = await User.findOne({ email: normalizedEmail });
        
        if (!group.members.includes(normalizedEmail)) group.members.push(normalizedEmail);
        
        // Create or update membership with user ID if user exists
        const membershipData = {
            group: groupId,
            email: normalizedEmail,
            role: 'member',
            canViewAllTasks: group.canMembersViewAllTasks
        };
        
        if (existingUser) {
            membershipData.user = existingUser._id;
        }
        
        await GroupMembership.updateOne(
            { group: groupId, email: normalizedEmail },
            { $setOnInsert: membershipData },
            { upsert: true }
        );
        
        await group.save();
        try {
            await emailService.sendMail({
                to: normalizedEmail,
                subject: `Added to group: ${group.name}`,
                text: `You have been added to the group "${group.name}".`
            });
        } catch (_) {}
        res.json({ message: 'Member added.' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Get group activity logs
exports.getGroupActivityLogs = async (req, res) => {
    try {
        const { groupId } = req.params;
        const { limit = 50, startDate, endDate } = req.query;
        
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found' });
        
        // Check if user is a member
        const userEmail = (req.user.email || '').trim().toLowerCase();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!membership) {
            return res.status(403).json({ error: 'Only group members can view activity logs.' });
        }
        
        // Build query
        const query = { groupId: groupId };
        if (startDate && endDate) {
            query.date = {
                $gte: new Date(startDate),
                $lte: new Date(endDate)
            };
        }
        
        const activities = await ActivityLog.find(query)
            .populate('userId', 'name email')
            .sort({ date: -1 })
            .limit(parseInt(limit));
        
        res.json(activities);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Get enhanced group analytics
exports.getGroupAnalytics = async (req, res) => {
    try {
        const { groupId } = req.params;
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found' });
        
        // Check if user is a member
        const userEmail = (req.user.email || '').trim().toLowerCase();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!membership) {
            return res.status(403).json({ error: 'Only group members can view analytics.' });
        }
        
        const tasks = await GroupTask.find({ group: groupId });
        const memberships = await GroupMembership.find({ group: groupId }).populate('user', 'name email');
        
        // Basic stats
        const totalTasks = tasks.length;
        const completedTasks = tasks.filter(t => t.completed || t.status === 'completed').length;
        const pendingTasks = totalTasks - completedTasks;
        const completionRate = totalTasks > 0 ? ((completedTasks / totalTasks) * 100).toFixed(1) : 0;
        
        // Priority distribution
        const priorityDistribution = {
            high: tasks.filter(t => t.priority === 'high').length,
            medium: tasks.filter(t => t.priority === 'medium').length,
            low: tasks.filter(t => t.priority === 'low').length
        };
        
        // Tasks by status
        const tasksByStatus = {
            completed: completedTasks,
            pending: pendingTasks,
            overdue: tasks.filter(t => {
                if (!t.dueDate || t.completed) return false;
                return new Date(t.dueDate) < new Date();
            }).length
        };
        
        // Member statistics
        const memberStats = [];
        for (const membership of memberships) {
            const memberEmail = membership.email;
            const memberTasks = tasks.filter(t => t.assignedTo?.includes(memberEmail));
            const memberCompleted = memberTasks.filter(t => t.completed || t.status === 'completed').length;
            const memberPoints = memberTasks
                .filter(t => t.completed || t.status === 'completed')
                .reduce((sum, t) => sum + (t.points || 0), 0);
            
            memberStats.push({
                email: memberEmail,
                name: membership.user?.name || memberEmail,
                totalTasks: memberTasks.length,
                completedTasks: memberCompleted,
                pendingTasks: memberTasks.length - memberCompleted,
                totalPoints: memberPoints,
                completionRate: memberTasks.length > 0 ? ((memberCompleted / memberTasks.length) * 100).toFixed(1) : 0
            });
        }
        
        // Tasks over time (last 30 days)
        const thirtyDaysAgo = new Date();
        thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
        const recentTasks = tasks.filter(t => new Date(t.createdAt) >= thirtyDaysAgo);
        const tasksOverTime = [];
        for (let i = 29; i >= 0; i--) {
            const date = new Date();
            date.setDate(date.getDate() - i);
            date.setHours(0, 0, 0, 0);
            const nextDate = new Date(date);
            nextDate.setDate(nextDate.getDate() + 1);
            
            const dayTasks = recentTasks.filter(t => {
                const taskDate = new Date(t.createdAt);
                return taskDate >= date && taskDate < nextDate;
            });
            
            tasksOverTime.push({
                date: date.toISOString().split('T')[0],
                count: dayTasks.length,
                completed: dayTasks.filter(t => t.completed || t.status === 'completed').length
            });
        }
        
        // Leaderboard
        const leaderboard = memberStats
            .sort((a, b) => b.totalPoints - a.totalPoints)
            .slice(0, 10);
        
        res.json({
            overview: {
                totalTasks,
                completedTasks,
                pendingTasks,
                completionRate: parseFloat(completionRate),
                totalMembers: memberships.length
            },
            priorityDistribution,
            tasksByStatus,
            memberStats,
            leaderboard,
            tasksOverTime
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Get advanced analytics for a group (comprehensive reports) - Premium only
exports.getAdvancedAnalytics = async (req, res) => {
    try {
        const { groupId } = req.params;
        const { period = 'monthly' } = req.query; // daily, weekly, monthly
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found' });
        
        // Check if user is a member
        const userEmail = (req.user.email || '').trim().toLowerCase();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!membership) {
            return res.status(403).json({ error: 'Only group members can view analytics.' });
        }
        
        // Premium check is handled by requirePremium middleware, but we add a note here
        
        const tasks = await GroupTask.find({ group: groupId });
        const memberships = await GroupMembership.find({ group: groupId }).populate('user', 'name email');
        const activities = await ActivityLog.find({ groupId: groupId }).sort({ date: -1 });
        
        // Calculate date ranges
        const now = new Date();
        const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
        const startOfWeek = new Date(today);
        startOfWeek.setDate(today.getDate() - today.getDay());
        const startOfMonth = new Date(now.getFullYear(), now.getMonth(), 1);
        const startOfYear = new Date(now.getFullYear(), 0, 1);
        
        // Helper function to get date range based on period
        const getDateRange = (periodType) => {
            switch(periodType) {
                case 'daily':
                    return { start: today, end: new Date(today.getTime() + 24 * 60 * 60 * 1000) };
                case 'weekly':
                    return { start: startOfWeek, end: now };
                case 'monthly':
                    return { start: startOfMonth, end: now };
                default:
                    return { start: startOfMonth, end: now };
            }
        };
        
        const dateRange = getDateRange(period);
        
        // Filter tasks by date range
        const periodTasks = tasks.filter(t => {
            const taskDate = new Date(t.createdAt);
            return taskDate >= dateRange.start && taskDate <= dateRange.end;
        });
        
        // Group Output Statistics
        const groupOutput = {
            totalTasksCreated: periodTasks.length,
            totalTasksCompleted: periodTasks.filter(t => t.completed || t.status === 'completed').length,
            totalPointsEarned: periodTasks
                .filter(t => t.completed || t.status === 'completed')
                .reduce((sum, t) => sum + (t.points || 0), 0),
            averageCompletionTime: calculateAverageCompletionTime(periodTasks),
            productivityScore: calculateProductivityScore(periodTasks, activities, dateRange),
            taskVelocity: calculateTaskVelocity(periodTasks, dateRange),
            qualityScore: calculateQualityScore(periodTasks, activities)
        };
        
        // User Performance Reports
        const userReports = [];
        for (const membership of memberships) {
            const memberEmail = membership.email;
            const memberTasks = periodTasks.filter(t => t.assignedTo?.includes(memberEmail));
            const memberCompleted = memberTasks.filter(t => t.completed || t.status === 'completed');
            const memberActivities = activities.filter(a => 
                a.userId?.email === memberEmail || 
                (typeof a.userId === 'string' && a.userId === membership.user?._id?.toString())
            );
            
            // Daily breakdown
            const dailyBreakdown = getDailyBreakdown(memberTasks, memberCompleted, dateRange);
            
            // Weekly breakdown
            const weeklyBreakdown = getWeeklyBreakdown(memberTasks, memberCompleted, dateRange);
            
            // Monthly breakdown
            const monthlyBreakdown = getMonthlyBreakdown(memberTasks, memberCompleted);
            
            userReports.push({
                email: memberEmail,
                name: membership.user?.name || memberEmail,
                userId: membership.user?._id || null,
                role: membership.role,
                // Overall stats
                totalTasks: memberTasks.length,
                completedTasks: memberCompleted.length,
                pendingTasks: memberTasks.length - memberCompleted.length,
                completionRate: memberTasks.length > 0 ? 
                    ((memberCompleted.length / memberTasks.length) * 100).toFixed(1) : 0,
                totalPoints: memberCompleted.reduce((sum, t) => sum + (t.points || 0), 0),
                // Time-based reports
                daily: dailyBreakdown,
                weekly: weeklyBreakdown,
                monthly: monthlyBreakdown,
                // Activity metrics
                activitiesCount: memberActivities.length,
                lastActive: memberActivities[0]?.date || null,
                // Performance metrics
                averageTaskCompletionTime: calculateAverageCompletionTime(memberCompleted),
                onTimeCompletionRate: calculateOnTimeCompletionRate(memberCompleted),
                productivityTrend: calculateProductivityTrend(memberTasks, dateRange)
            });
        }
        
        // Time-based Analysis
        const timeAnalysis = {
            tasksOverTime: getTasksOverTime(periodTasks, dateRange, period),
            completionTrend: getCompletionTrend(periodTasks, dateRange, period),
            activityHeatmap: getActivityHeatmap(activities, dateRange),
            peakProductivityHours: getPeakProductivityHours(activities, dateRange)
        };
        
        // Section Performance
        const sections = await GroupSection.find({ group: groupId });
        const sectionPerformance = sections.map(section => {
            const sectionTasks = periodTasks.filter(t => {
                const taskSectionId = t.sectionId?._id || t.sectionId;
                return taskSectionId && taskSectionId.toString() === section._id.toString();
            });
            const sectionCompleted = sectionTasks.filter(t => t.completed || t.status === 'completed');
            
            return {
                sectionId: section._id,
                sectionName: section.name,
                totalTasks: sectionTasks.length,
                completedTasks: sectionCompleted.length,
                completionRate: sectionTasks.length > 0 ? 
                    ((sectionCompleted.length / sectionTasks.length) * 100).toFixed(1) : 0,
                totalPoints: sectionCompleted.reduce((sum, t) => sum + (t.points || 0), 0)
            };
        });
        
        // Priority Analysis
        const priorityAnalysis = {
            high: {
                total: periodTasks.filter(t => t.priority === 'high').length,
                completed: periodTasks.filter(t => t.priority === 'high' && (t.completed || t.status === 'completed')).length,
                averageCompletionTime: calculateAverageCompletionTime(
                    periodTasks.filter(t => t.priority === 'high' && (t.completed || t.status === 'completed'))
                )
            },
            medium: {
                total: periodTasks.filter(t => t.priority === 'medium').length,
                completed: periodTasks.filter(t => t.priority === 'medium' && (t.completed || t.status === 'completed')).length,
                averageCompletionTime: calculateAverageCompletionTime(
                    periodTasks.filter(t => t.priority === 'medium' && (t.completed || t.status === 'completed'))
                )
            },
            low: {
                total: periodTasks.filter(t => t.priority === 'low').length,
                completed: periodTasks.filter(t => t.priority === 'low' && (t.completed || t.status === 'completed')).length,
                averageCompletionTime: calculateAverageCompletionTime(
                    periodTasks.filter(t => t.priority === 'low' && (t.completed || t.status === 'completed'))
                )
            }
        };
        
        res.json({
            period,
            dateRange: {
                start: dateRange.start.toISOString(),
                end: dateRange.end.toISOString()
            },
            groupOutput,
            userReports,
            timeAnalysis,
            sectionPerformance,
            priorityAnalysis,
            summary: {
                totalMembers: memberships.length,
                activeMembers: userReports.filter(u => u.activitiesCount > 0).length,
                totalTasks: periodTasks.length,
                completedTasks: periodTasks.filter(t => t.completed || t.status === 'completed').length,
                overallCompletionRate: periodTasks.length > 0 ? 
                    ((periodTasks.filter(t => t.completed || t.status === 'completed').length / periodTasks.length) * 100).toFixed(1) : 0
            }
        });
    } catch (err) {
        console.error('Error in getAdvancedAnalytics:', err);
        res.status(500).json({ error: err.message });
    }
};

// Helper functions for advanced analytics
function calculateAverageCompletionTime(completedTasks) {
    if (!completedTasks || completedTasks.length === 0) return 0;
    
    const times = completedTasks
        .filter(t => t.createdAt && t.updatedAt && (t.completed || t.status === 'completed'))
        .map(t => {
            const created = new Date(t.createdAt);
            const completed = new Date(t.updatedAt);
            return (completed - created) / (1000 * 60 * 60); // hours
        })
        .filter(t => t > 0 && t < 10000); // reasonable range
    
    if (times.length === 0) return 0;
    return (times.reduce((a, b) => a + b, 0) / times.length).toFixed(1);
}

function calculateProductivityScore(tasks, activities, dateRange) {
    const completedTasks = tasks.filter(t => t.completed || t.status === 'completed').length;
    const totalTasks = tasks.length;
    const activityCount = activities.filter(a => {
        const activityDate = new Date(a.date || a.createdAt);
        return activityDate >= dateRange.start && activityDate <= dateRange.end;
    }).length;
    
    const taskScore = totalTasks > 0 ? (completedTasks / totalTasks) * 50 : 0;
    const activityScore = Math.min(activityCount / 10, 50); // cap at 50
    
    return (taskScore + activityScore).toFixed(1);
}

function calculateTaskVelocity(tasks, dateRange) {
    const daysDiff = Math.max(1, Math.ceil((dateRange.end - dateRange.start) / (1000 * 60 * 60 * 24)));
    const completedTasks = tasks.filter(t => t.completed || t.status === 'completed').length;
    return (completedTasks / daysDiff).toFixed(2);
}

function calculateQualityScore(tasks, activities) {
    // Based on completion rate and activity engagement
    const completedTasks = tasks.filter(t => t.completed || t.status === 'completed').length;
    const completionRate = tasks.length > 0 ? (completedTasks / tasks.length) * 100 : 0;
    const activityEngagement = Math.min(activities.length / 20, 100); // normalize
    
    return ((completionRate * 0.7) + (activityEngagement * 0.3)).toFixed(1);
}

function getDailyBreakdown(tasks, completedTasks, dateRange) {
    const breakdown = {};
    const daysDiff = Math.ceil((dateRange.end - dateRange.start) / (1000 * 60 * 60 * 24));
    
    for (let i = 0; i < daysDiff; i++) {
        const date = new Date(dateRange.start);
        date.setDate(date.getDate() + i);
        const dateStr = date.toISOString().split('T')[0];
        
        const dayTasks = tasks.filter(t => {
            const taskDate = new Date(t.createdAt).toISOString().split('T')[0];
            return taskDate === dateStr;
        });
        
        const dayCompleted = completedTasks.filter(t => {
            const taskDate = new Date(t.updatedAt || t.createdAt).toISOString().split('T')[0];
            return taskDate === dateStr;
        });
        
        breakdown[dateStr] = {
            created: dayTasks.length,
            completed: dayCompleted.length,
            points: dayCompleted.reduce((sum, t) => sum + (t.points || 0), 0)
        };
    }
    
    return breakdown;
}

function getWeeklyBreakdown(tasks, completedTasks, dateRange) {
    const breakdown = {};
    const weeks = [];
    let currentWeek = new Date(dateRange.start);
    
    while (currentWeek <= dateRange.end) {
        const weekStart = new Date(currentWeek);
        weekStart.setDate(currentWeek.getDate() - currentWeek.getDay());
        const weekEnd = new Date(weekStart);
        weekEnd.setDate(weekStart.getDate() + 6);
        
        const weekTasks = tasks.filter(t => {
            const taskDate = new Date(t.createdAt);
            return taskDate >= weekStart && taskDate <= weekEnd;
        });
        
        const weekCompleted = completedTasks.filter(t => {
            const taskDate = new Date(t.updatedAt || t.createdAt);
            return taskDate >= weekStart && taskDate <= weekEnd;
        });
        
        const weekKey = `${weekStart.toISOString().split('T')[0]}_${weekEnd.toISOString().split('T')[0]}`;
        breakdown[weekKey] = {
            created: weekTasks.length,
            completed: weekCompleted.length,
            points: weekCompleted.reduce((sum, t) => sum + (t.points || 0), 0)
        };
        
        currentWeek.setDate(currentWeek.getDate() + 7);
    }
    
    return breakdown;
}

function getMonthlyBreakdown(tasks, completedTasks) {
    const breakdown = {};
    
    tasks.forEach(task => {
        const taskDate = new Date(task.createdAt);
        const monthKey = `${taskDate.getFullYear()}-${String(taskDate.getMonth() + 1).padStart(2, '0')}`;
        
        if (!breakdown[monthKey]) {
            breakdown[monthKey] = { created: 0, completed: 0, points: 0 };
        }
        breakdown[monthKey].created++;
    });
    
    completedTasks.forEach(task => {
        const taskDate = new Date(task.updatedAt || task.createdAt);
        const monthKey = `${taskDate.getFullYear()}-${String(taskDate.getMonth() + 1).padStart(2, '0')}`;
        
        if (breakdown[monthKey]) {
            breakdown[monthKey].completed++;
            breakdown[monthKey].points += (task.points || 0);
        }
    });
    
    return breakdown;
}

function calculateOnTimeCompletionRate(completedTasks) {
    if (!completedTasks || completedTasks.length === 0) return 0;
    
    const onTime = completedTasks.filter(t => {
        if (!t.dueDate) return false;
        const completed = new Date(t.updatedAt || t.createdAt);
        const due = new Date(t.dueDate);
        return completed <= due;
    }).length;
    
    return ((onTime / completedTasks.length) * 100).toFixed(1);
}

function calculateProductivityTrend(tasks, dateRange) {
    const daysDiff = Math.ceil((dateRange.end - dateRange.start) / (1000 * 60 * 60 * 24));
    if (daysDiff < 2) return 'stable';
    
    const firstHalf = tasks.filter(t => {
        const taskDate = new Date(t.createdAt);
        const midPoint = new Date(dateRange.start.getTime() + (dateRange.end - dateRange.start) / 2);
        return taskDate < midPoint;
    }).length;
    
    const secondHalf = tasks.length - firstHalf;
    
    if (secondHalf > firstHalf * 1.2) return 'increasing';
    if (secondHalf < firstHalf * 0.8) return 'decreasing';
    return 'stable';
}

function getTasksOverTime(tasks, dateRange, period) {
    const data = [];
    const interval = period === 'daily' ? 1 : period === 'weekly' ? 7 : 30;
    let current = new Date(dateRange.start);
    
    while (current <= dateRange.end) {
        const next = new Date(current);
        next.setDate(next.getDate() + interval);
        
        const intervalTasks = tasks.filter(t => {
            const taskDate = new Date(t.createdAt);
            return taskDate >= current && taskDate < next;
        });
        
        data.push({
            date: current.toISOString().split('T')[0],
            created: intervalTasks.length,
            completed: intervalTasks.filter(t => t.completed || t.status === 'completed').length
        });
        
        current = next;
    }
    
    return data;
}

function getCompletionTrend(tasks, dateRange, period) {
    return getTasksOverTime(tasks, dateRange, period).map(d => ({
        date: d.date,
        completionRate: d.created > 0 ? ((d.completed / d.created) * 100).toFixed(1) : 0
    }));
}

function getActivityHeatmap(activities, dateRange) {
    const heatmap = {};
    activities.forEach(activity => {
        const activityDate = new Date(activity.date || activity.createdAt);
        if (activityDate >= dateRange.start && activityDate <= dateRange.end) {
            const dateStr = activityDate.toISOString().split('T')[0];
            heatmap[dateStr] = (heatmap[dateStr] || 0) + 1;
        }
    });
    return heatmap;
}

function getPeakProductivityHours(activities, dateRange) {
    const hourCounts = Array(24).fill(0);
    activities.forEach(activity => {
        const activityDate = new Date(activity.date || activity.createdAt);
        if (activityDate >= dateRange.start && activityDate <= dateRange.end) {
            const hour = activityDate.getHours();
            hourCounts[hour]++;
        }
    });
    
    const maxCount = Math.max(...hourCounts);
    const peakHours = hourCounts.map((count, hour) => ({
        hour,
        count,
        percentage: maxCount > 0 ? ((count / maxCount) * 100).toFixed(1) : 0
    })).sort((a, b) => b.count - a.count);
    
    return peakHours;
}

// Get all sections for a group
exports.getGroupSections = async (req, res) => {
    try {
        const { groupId } = req.params;
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found' });
        
        // Check if user is a member
        const userEmail = (req.user.email || '').trim().toLowerCase();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!membership) {
            return res.status(403).json({ error: 'Only group members can view sections.' });
        }
        
        const sections = await GroupSection.find({ groupId })
            .populate('createdBy', 'name email')
            .sort({ createdAt: -1 });
        
        res.json(sections);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Create a new section for a group
exports.createGroupSection = async (req, res) => {
    try {
        const { groupId } = req.params;
        const { name, description } = req.body;
        
        if (!name) return res.status(400).json({ error: 'Section name is required.' });
        
        const group = await Group.findById(groupId);
        if (!group) return res.status(404).json({ error: 'Group not found' });
        
        // Check if user is a member
        const userEmail = (req.user.email || '').trim().toLowerCase();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!membership) {
            return res.status(403).json({ error: 'Only group members can create sections.' });
        }
        
        const section = await GroupSection.create({
            groupId,
            name,
            description: description || '',
            createdBy: req.user._id
        });
        
        // Log activity
        try {
            await ActivityLog.logActivity({
                userId: req.user._id,
                action: 'section_created',
                groupId: groupId,
                groupName: group.name,
                metadata: { sectionName: name }
            });
        } catch (logErr) {
            console.error('Error logging activity:', logErr);
        }
        
        const populatedSection = await GroupSection.findById(section._id)
            .populate('createdBy', 'name email');
        
        res.status(201).json(populatedSection);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Update a group section
exports.updateGroupSection = async (req, res) => {
    try {
        const { groupId, sectionId } = req.params;
        const { name, description } = req.body;
        
        const section = await GroupSection.findById(sectionId);
        if (!section || section.groupId.toString() !== groupId) {
            return res.status(404).json({ error: 'Section not found' });
        }
        
        // Check if user is a member
        const userEmail = (req.user.email || '').trim().toLowerCase();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!membership) {
            return res.status(403).json({ error: 'Only group members can update sections.' });
        }
        
        if (name !== undefined) section.name = name;
        if (description !== undefined) section.description = description;
        
        await section.save();
        
        const populatedSection = await GroupSection.findById(section._id)
            .populate('createdBy', 'name email');
        
        res.json(populatedSection);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// Delete a group section
exports.deleteGroupSection = async (req, res) => {
    try {
        const { groupId, sectionId } = req.params;
        
        const section = await GroupSection.findById(sectionId);
        if (!section || section.groupId.toString() !== groupId) {
            return res.status(404).json({ error: 'Section not found' });
        }
        
        // Check if user is a member and either creator or admin
        const userEmail = (req.user.email || '').trim().toLowerCase();
        const membership = await GroupMembership.findOne({ group: groupId, email: userEmail });
        if (!membership) {
            return res.status(403).json({ error: 'Only group members can delete sections.' });
        }
        
        const isCreator = section.createdBy.toString() === req.user._id.toString();
        const isAdmin = membership.role === 'creator' || membership.role === 'admin';
        
        if (!isCreator && !isAdmin) {
            return res.status(403).json({ error: 'Only section creator or group admin can delete sections.' });
        }
        
        await GroupSection.findByIdAndDelete(sectionId);
        
        // Log activity
        try {
            const group = await Group.findById(groupId);
            await ActivityLog.logActivity({
                userId: req.user._id,
                action: 'section_created',
                groupId: groupId,
                groupName: group?.name,
                metadata: { sectionName: section.name, action: 'deleted' }
            });
        } catch (logErr) {
            console.error('Error logging activity:', logErr);
        }
        
        res.json({ message: 'Section deleted successfully.' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};