const Course = require('../models/Course');
const CourseSubscription = require('../models/CourseSubscription');
const Assignment = require('../models/Assignment');

// Create course
exports.createCourse = async (req, res) => {
    try {
        const { title, description, price, isPremium, category, tags, thumbnail } = req.body;
        const userId = req.user._id;

        // Only admins can create courses for now (can be extended)
        if (req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Only admins can create courses' });
        }

        const course = new Course({
            title,
            description,
            instructor: userId,
            price: price || 0,
            isPremium: isPremium || false,
            category: category || 'general',
            tags: tags || [],
            thumbnail: thumbnail || null,
            status: 'published'
        });

        await course.save();
        res.status(201).json(course);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Get all courses
exports.getAllCourses = async (req, res) => {
    try {
        const { category, search, isPremium } = req.query;
        const userId = req.user._id;

        const query = { status: 'published' };
        
        if (category) query.category = category;
        if (isPremium !== undefined) query.isPremium = isPremium === 'true';
        if (search) {
            query.$or = [
                { title: { $regex: search, $options: 'i' } },
                { description: { $regex: search, $options: 'i' } }
            ];
        }

        const courses = await Course.find(query)
            .populate('instructor', 'name email')
            .sort({ createdAt: -1 });

        // Add enrollment status for each course
        const coursesWithEnrollment = await Promise.all(
            courses.map(async (course) => {
                const courseObj = course.toObject();
                const subscription = await CourseSubscription.findOne({
                    user: userId,
                    course: course._id
                });
                courseObj.isEnrolled = !!subscription;
                courseObj.enrollmentStatus = subscription?.status || null;
                return courseObj;
            })
        );

        res.json(coursesWithEnrollment);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Get single course
exports.getCourse = async (req, res) => {
    try {
        const { courseId } = req.params;
        const userId = req.user._id;

        const course = await Course.findById(courseId)
            .populate('instructor', 'name email')
            .populate('assignments');

        if (!course) {
            return res.status(404).json({ error: 'Course not found' });
        }

        const courseObj = course.toObject();
        
        // Check enrollment
        const subscription = await CourseSubscription.findOne({
            user: userId,
            course: courseId
        });
        
        courseObj.isEnrolled = !!subscription;
        courseObj.enrollmentStatus = subscription?.status || null;
        courseObj.progress = subscription?.progress || 0;

        res.json(courseObj);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Subscribe to course
exports.subscribeToCourse = async (req, res) => {
    try {
        const { courseId } = req.params;
        const userId = req.user._id;

        const course = await Course.findById(courseId);
        if (!course) {
            return res.status(404).json({ error: 'Course not found' });
        }

        // Check if already subscribed
        const existingSubscription = await CourseSubscription.findOne({
            user: userId,
            course: courseId
        });

        if (existingSubscription) {
            if (existingSubscription.status === 'active') {
                return res.status(400).json({ error: 'Already subscribed to this course' });
            } else {
                // Reactivate subscription
                existingSubscription.status = 'active';
                await existingSubscription.save();
                return res.json(existingSubscription);
            }
        }

        // Create new subscription
        const subscription = new CourseSubscription({
            user: userId,
            course: courseId
        });

        await subscription.save();

        // Add to course enrolled students
        course.enrolledStudents.push({
            userId: userId,
            enrolledAt: new Date(),
            progress: 0
        });
        await course.save();

        res.status(201).json(subscription);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Get user's enrolled courses
exports.getMyCourses = async (req, res) => {
    try {
        const userId = req.user._id;

        const subscriptions = await CourseSubscription.find({
            user: userId,
            status: 'active'
        }).populate({
            path: 'course',
            populate: {
                path: 'instructor',
                select: 'name email'
            }
        });

        const courses = subscriptions.map(sub => ({
            ...sub.course.toObject(),
            enrolledAt: sub.enrolledAt,
            progress: sub.progress,
            subscriptionId: sub._id
        }));

        res.json(courses);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Update course
exports.updateCourse = async (req, res) => {
    try {
        const { courseId } = req.params;
        const userId = req.user._id;

        const course = await Course.findById(courseId);
        if (!course) {
            return res.status(404).json({ error: 'Course not found' });
        }

        if (course.instructor.toString() !== userId.toString() && req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Only instructor can update course' });
        }

        Object.assign(course, req.body);
        await course.save();

        res.json(course);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Delete course
exports.deleteCourse = async (req, res) => {
    try {
        const { courseId } = req.params;
        const userId = req.user._id;

        const course = await Course.findById(courseId);
        if (!course) {
            return res.status(404).json({ error: 'Course not found' });
        }

        if (course.instructor.toString() !== userId.toString() && req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Only instructor can delete course' });
        }

        // Delete all assignments
        await Assignment.deleteMany({ course: courseId });

        // Delete all subscriptions
        await CourseSubscription.deleteMany({ course: courseId });

        await Course.findByIdAndDelete(courseId);

        res.json({ message: 'Course deleted successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

