const Coupon = require('../models/Coupon');

/**
 * Get all coupons (admin only)
 */
const getAllCoupons = async (req, res) => {
    try {
        const coupons = await Coupon.find()
            .populate('usedBy.user', 'name email')
            .sort({ createdAt: -1 });

        res.status(200).json({
            success: true,
            coupons
        });
    } catch (error) {
        console.error('Error fetching coupons:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching coupons',
            error: error.message
        });
    }
};

/**
 * Get active coupons
 */
const getActiveCoupons = async (req, res) => {
    try {
        const coupons = await Coupon.find({
            active: true,
            expiryDate: { $gte: new Date() }
        }).sort({ createdAt: -1 });

        res.status(200).json({
            success: true,
            coupons
        });
    } catch (error) {
        console.error('Error fetching active coupons:', error);
        res.status(500).json({
            success: false,
            message: 'Error fetching active coupons',
            error: error.message
        });
    }
};

/**
 * Validate coupon code
 */
const validateCoupon = async (req, res) => {
    try {
        const { code, planId } = req.body;

        if (!code) {
            return res.status(400).json({
                success: false,
                message: 'Coupon code is required'
            });
        }

        const coupon = await Coupon.findOne({
            code: code.toUpperCase().trim(),
            active: true,
            expiryDate: { $gte: new Date() }
        });

        if (!coupon) {
            return res.status(404).json({
                success: false,
                message: 'Invalid or expired coupon code',
                valid: false
            });
        }

        // Check if user already used this coupon
        const userId = req.user?._id;
        if (userId) {
            const alreadyUsed = coupon.usedBy.some(u => u.user && u.user.toString() === userId.toString());
            if (alreadyUsed) {
                return res.status(400).json({
                    success: false,
                    message: 'Coupon already used',
                    valid: false
                });
            }
        }

        // Check max uses
        if (coupon.usedBy.length >= coupon.maxUses) {
            return res.status(400).json({
                success: false,
                message: 'Coupon has reached maximum uses',
                valid: false
            });
        }

        // Check if applicable to plan
        if (planId && coupon.applicablePlans.length > 0 && !coupon.applicablePlans.includes(planId)) {
            return res.status(400).json({
                success: false,
                message: 'Coupon not applicable to selected plan',
                valid: false
            });
        }

        res.status(200).json({
            success: true,
            valid: true,
            coupon: {
                code: coupon.code,
                discountType: coupon.discountType,
                value: coupon.value,
                description: coupon.description
            }
        });
    } catch (error) {
        console.error('Error validating coupon:', error);
        res.status(500).json({
            success: false,
            message: 'Error validating coupon',
            error: error.message
        });
    }
};

/**
 * Create coupon (admin only)
 */
const createCoupon = async (req, res) => {
    try {
        const { code, discountType, value, applicablePlans, expiryDate, maxUses, description } = req.body;

        if (!code || !discountType || value === undefined || !expiryDate) {
            return res.status(400).json({
                success: false,
                message: 'Missing required fields: code, discountType, value, expiryDate'
            });
        }

        if (discountType === 'percentage' && (value < 0 || value > 100)) {
            return res.status(400).json({
                success: false,
                message: 'Percentage discount must be between 0 and 100'
            });
        }

        // Check if code already exists
        const existingCoupon = await Coupon.findOne({ code: code.toUpperCase().trim() });
        if (existingCoupon) {
            return res.status(400).json({
                success: false,
                message: 'Coupon code already exists'
            });
        }

        const coupon = new Coupon({
            code: code.toUpperCase().trim(),
            discountType,
            value,
            applicablePlans: applicablePlans || [],
            expiryDate: new Date(expiryDate),
            maxUses: maxUses || 1,
            description
        });

        await coupon.save();

        res.status(201).json({
            success: true,
            message: 'Coupon created successfully',
            coupon
        });
    } catch (error) {
        console.error('Error creating coupon:', error);
        if (error.code === 11000) {
            return res.status(400).json({
                success: false,
                message: 'Coupon code already exists'
            });
        }
        res.status(500).json({
            success: false,
            message: 'Error creating coupon',
            error: error.message
        });
    }
};

/**
 * Update coupon (admin only)
 */
const updateCoupon = async (req, res) => {
    try {
        const { couponId } = req.params;
        const updates = req.body;

        if (updates.value !== undefined && updates.discountType === 'percentage' && (updates.value < 0 || updates.value > 100)) {
            return res.status(400).json({
                success: false,
                message: 'Percentage discount must be between 0 and 100'
            });
        }

        if (updates.expiryDate) updates.expiryDate = new Date(updates.expiryDate);
        if (updates.code) updates.code = updates.code.toUpperCase().trim();

        const coupon = await Coupon.findByIdAndUpdate(
            couponId,
            { $set: updates },
            { new: true, runValidators: true }
        );

        if (!coupon) {
            return res.status(404).json({
                success: false,
                message: 'Coupon not found'
            });
        }

        res.status(200).json({
            success: true,
            message: 'Coupon updated successfully',
            coupon
        });
    } catch (error) {
        console.error('Error updating coupon:', error);
        res.status(500).json({
            success: false,
            message: 'Error updating coupon',
            error: error.message
        });
    }
};

/**
 * Delete coupon (admin only)
 */
const deleteCoupon = async (req, res) => {
    try {
        const { couponId } = req.params;

        const coupon = await Coupon.findByIdAndDelete(couponId);

        if (!coupon) {
            return res.status(404).json({
                success: false,
                message: 'Coupon not found'
            });
        }

        res.status(200).json({
            success: true,
            message: 'Coupon deleted successfully'
        });
    } catch (error) {
        console.error('Error deleting coupon:', error);
        res.status(500).json({
            success: false,
            message: 'Error deleting coupon',
            error: error.message
        });
    }
};

module.exports = {
    getAllCoupons,
    getActiveCoupons,
    validateCoupon,
    createCoupon,
    updateCoupon,
    deleteCoupon
};

