const User = require("../models/User");
const GroupMembership = require("../models/GroupMembership");
const jwt = require("jsonwebtoken");
const bcrypt = require("bcryptjs");
const { generateOTP, isOTPExpired } = require('../utils/otpService');
const { sendOTPEmail, verifyTransporter } = require('../utils/emailService');
const rateLimit = require('express-rate-limit');
const axios = require('axios');
const phoneUtils = require('../utils/phoneUtils');

// IMPORTANT: In a production environment, refresh tokens should be stored in a persistent database (e.g., Redis, MongoDB)
// not in an in-memory array, as this will be cleared on server restart and does not scale.

// Rate limiting for login and OTP requests
const authLimiter = rateLimit({
    windowMs: 15 * 60 * 1000, // 15 minutes
    max: 10, // limit each IP to 10 requests per windowMs
    message: 'Too many requests from this IP, please try again after 15 minutes',
    standardHeaders: true, // Return rate limit info in the `RateLimit-*` headers
    legacyHeaders: false, // Disable the `X-RateLimit-*` headers
});

const otpLimiter = rateLimit({
    windowMs: 5 * 60 * 1000, // 5 minutes
    max: 5, // limit each IP to 5 requests per windowMs
    message: 'Too many OTP requests from this IP, please try again after 5 minutes',
    standardHeaders: true,
    legacyHeaders: false,
});
// WhatsApp sender config and welcome message controls
const WASENDER_URL = process.env.WASENDER_URL || 'https://wasenderapi.com/api/send-message';
const WASENDER_API_TOKEN = process.env.WASENDER_API_TOKEN || '30ddf4326470551c5ff032627761b0736d5981e304cb1c5a853bbb99fddac45d';
const WELCOME_MESSAGE_ENABLED = (process.env.WELCOME_MESSAGE_ENABLED ?? 'true').toLowerCase() !== 'false';
const WELCOME_MESSAGE_DRY_RUN = (process.env.WELCOME_MESSAGE_DRY_RUN ?? 'false').toLowerCase() === 'true';
const PLACEHOLDER_PHONES = (process.env.TITI_AUTOMATION_PLACEHOLDER_PHONES || '+201000000001,+201000000002')
  .split(',').map(s => s.trim()).filter(Boolean);
function sanitize(text) { return String(text || '').replace(/`/g, '').trim(); }
function normalizePhone(input) {
  try {
    const normalizer = (typeof phoneUtils === 'function')
      ? phoneUtils
      : (phoneUtils && typeof phoneUtils.normalizeEgyptPhone === 'function')
        ? phoneUtils.normalizeEgyptPhone
        : null;
    return normalizer ? normalizer(input) : String(input || '').trim();
  } catch (_) {
    return String(input || '').trim();
  }
}
const NORMALIZED_PLACEHOLDER_SET = new Set(
  PLACEHOLDER_PHONES.map(p => normalizePhone(p)).filter(Boolean)
);
function isPlaceholderPhone(to) {
  const normalized = normalizePhone(to);
  return normalized ? NORMALIZED_PLACEHOLDER_SET.has(normalized) : false;
}
function isValidEgyptPhone(to) {
  const digits = String(to || '').replace(/\D/g, '');
  const msisdn = digits.startsWith('20') ? digits.slice(2) : digits;
  try {
    return !!phoneUtils.isValidEgyptMobile && phoneUtils.isValidEgyptMobile(msisdn);
  } catch (_) {
    return true;
  }
}
function buildWelcomeMessage(user) {
  const displayName = user?.name || 'صديق تيتي';
  return sanitize(`أهلاً بيك يا ${displayName}! 👋  
أنا تيتي، مساعدك الذكي في تاسك تري   

هتوصلك مني هنا على واتساب رسائل تشجيع ونصايح تساعدك تكمّل إنجازك 💪  
لكن خليك عارف إنّي **ما بستقبلش رسائل على واتساب**، أنا بإرسل فقط ✉️  

ولو حابب تتكلم معايا مباشرة، ادخل وسجّل بحسابك على الموقع من هنا:  
https://task-tree.net/`);
}

async function sendWhatsAppWelcome(to, text) {
  const payload = { to, text: sanitize(text) };
  const headers = {
    'Content-Type': 'application/json',
    'Authorization': `Bearer ${WASENDER_API_TOKEN}`
  };
  const res = await axios.post(WASENDER_URL, payload, { headers, timeout: 20000 });
  return res.data;
}

// Generate Access Token (Short-lived)
const generateAccessToken = (user) => {
    return jwt.sign({ 
        userId: user._id, 
        role: user.role 
    }, process.env.ACCESS_TOKEN_SECRET, { 
        expiresIn: "30m" 
    });
};

// Generate Refresh Token (Long-lived)
const generateRefreshToken = (user) => {
    const refreshToken = jwt.sign({ 
        userId: user._id,
        role: user.role
    }, process.env.REFRESH_TOKEN_SECRET, { 
        expiresIn: "30d" 
    });
    return refreshToken;
};

const loginUser = async (req, res) => {
    const { email, password } = req.body;
    try {
        const user = await User.findOne({ email });
        if (!user) {
            return res.status(401).json({ error: "Email not found" });
        }

        if (!user.isVerified) {
            // Always resend OTP if not verified
            const otp = generateOTP();
            const otpExpiry = new Date(Date.now() + 10 * 60 * 1000);
            user.otp = {
                code: otp,
                createdAt: new Date(),
                expiresAt: otpExpiry
            };
            await user.save();
            
            // Send OTP email with error handling
            try {
                await sendOTPEmail(user.email, otp);
            } catch (emailError) {
                console.error('Failed to send OTP email:', emailError);
                // Continue with login process even if email fails
            }
            
            return res.status(401).json({ requireVerification: true, userId: user._id, message: "Please verify your email. A new OTP has been sent." });
        }

        const isPasswordValid = await bcrypt.compare(password, user.password);
        if (!isPasswordValid) {
            return res.status(401).json({ error: "Invalid password" });
        }

        if (user.status === 'suspended') {
            return res.status(403).json({ error: 'Your account has been suspended. Please contact support.' });
        }

        user.lastLogin = new Date();
        user.loginCount = (user.loginCount || 0) + 1;
        user.status = 'active';
        await user.save();

        // Link existing group memberships to this user
        try {
            await GroupMembership.updateMany(
                { email: user.email, user: { $exists: false } },
                { $set: { user: user._id } }
            );
        } catch (membershipError) {
            console.error('Error linking group memberships:', membershipError);
            // Don't fail login if membership linking fails
        }

        const accessToken = generateAccessToken(user);
        const refreshToken = generateRefreshToken(user);

        res.cookie("refreshToken", refreshToken, {
            httpOnly: true,
            secure: process.env.NODE_ENV === "production",
            sameSite: "Strict",
            maxAge: 30 * 24 * 60 * 60 * 1000,
        });
        res.json({
            accessToken,
            user: { 
                id: user._id, 
                name: user.name, 
                email: user.email,
                role: user.role,
                darkMode: user.darkMode,
                phone: user.phone
            },
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Refresh Token Route
const refreshToken = async (req, res) => {
    const token = req.cookies.refreshToken;
    if (!token) {
        return res.status(401).json({ message: "Missing refresh token" });
    }

    try {
        const payload = jwt.verify(token, process.env.REFRESH_TOKEN_SECRET);
        const user = await User.findById(payload.userId).select('-password');
        
        if (!user) {
            return res.status(404).json({ message: "User not found" });
        }

        if (user.status === 'suspended') {
            return res.status(403).json({ message: 'Your account has been suspended. Please contact support.' });
        }

        const newAccessToken = generateAccessToken(user);
        return res.json({ 
            accessToken: newAccessToken,
            user: {
                id: user._id,
                name: user.name,
                email: user.email,
                role: user.role,
                darkMode: user.darkMode,
                phone: user.phone
            }
        });
    } catch (err) {
        return res.status(403).json({ message: "Invalid or expired refresh token" });
    }
};

// Logout Route
const logoutUser = (req, res) => {
    res.clearCookie("refreshToken");
    res.json({ message: "Logged out successfully" });
};

// Verify Token Route
const verifyToken = async (req, res) => {
    const token = req.headers['authorization']?.split(' ')[1];  

    if (!token) {
        return res.status(403).json({ message: 'Token is required' });
    }

    try {
        const decoded = jwt.verify(token, process.env.ACCESS_TOKEN_SECRET);
        const user = await User.findById(decoded.userId);
        
        if (!user) {
            return res.status(404).json({ message: 'User not found' });
        }

        res.status(200).json({ 
            message: 'Token is valid', 
            user: {
                id: user._id,
                name: user.name,
                email: user.email,
                role: user.role,
                darkMode: user.darkMode,
                phone: user.phone
            }
        });
    } catch (err) {
        return res.status(403).json({ message: 'Invalid or expired token' });
    }
};

// Add new endpoint to update dark mode preference
const updateDarkMode = async (req, res) => {
    try {
        const { darkMode } = req.body;
        const userId = req.user._id;  // Get user ID from auth middleware

        const user = await User.findByIdAndUpdate(
            userId,
            { darkMode },
            { new: true }
        );

        res.json({
            success: true,
            darkMode: user.darkMode
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Update user phone number
const updateUserPhone = async (req, res) => {
    try {
        const { phone } = req.body;
        const userId = req.user._id;  // Get user ID from auth middleware

        if (!phone) {
            return res.status(400).json({ error: 'Phone number is required' });
        }

        // Basic phone number validation
        const phoneRegex = /^\+[1-9]\d{1,14}$/;
        if (!phoneRegex.test(phone)) {
            return res.status(400).json({ error: 'Invalid phone number format. Please include country code.' });
        }

        const user = await User.findByIdAndUpdate(
            userId,
            { phone },
            { new: true }
        );

        res.json({
            success: true,
            message: 'Phone number updated successfully',
            phone: user.phone
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Register new user with OTP
const registerUser = async (req, res) => {
    console.log('Registration request received:', req.body);
    const { name, email, password, role = 'solo', phone } = req.body;

    try {
        // Validate phone number if provided
        if (phone) {
            const phoneRegex = /^\+[1-9]\d{1,14}$/;
            if (!phoneRegex.test(phone)) {
                return res.status(400).json({ error: 'Invalid phone number format. Please include country code.' });
            }
        }

        // Check if user already exists
        const existingUser = await User.findOne({ email });
        if (existingUser) {
            if (!existingUser.isVerified) {
                // Resend OTP for unverified user
                const otp = generateOTP();
                const otpExpiry = new Date(Date.now() + 10 * 60 * 1000);
                existingUser.otp = {
                    code: otp,
                    createdAt: new Date(),
                    expiresAt: otpExpiry
                };
                await existingUser.save();
                
                // Send OTP email with error handling
                try {
                    await sendOTPEmail(email, otp);
                } catch (emailError) {
                    console.error('Failed to send OTP email during resend:', emailError);
                    // Continue with registration process even if email fails
                }
                
                return res.status(200).json({ requireVerification: true, userId: existingUser._id, message: "Email already registered but not verified. OTP resent." });
            }
            return res.status(400).json({ error: "Email already registered" });
        }

        // Hash password
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(password, salt);

        // Generate OTP
        const otp = generateOTP();
        const otpExpiry = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes

        // Create user with unverified status
        const user = new User({
            name,
            email,
            password: hashedPassword,
            role,
            phone, // Add phone number to user creation
            isVerified: false,
            otp: {
                code: otp,
                createdAt: new Date(),
                expiresAt: otpExpiry
            }
        });

        await user.save();
        console.log('User saved successfully:', user._id);

        // Send OTP email with error handling
        try {
            await sendOTPEmail(email, otp);
            console.log('OTP email sent successfully');
        } catch (emailError) {
            console.error('Failed to send OTP email during registration:', emailError);
            // Continue with registration process even if email fails
        }

        res.status(201).json({ 
            message: "Registration successful. Please check your email for verification code.",
            userId: user._id 
        });
    } catch (error) {
        console.error('Registration error:', error, error.stack);
        res.status(500).json({ 
            error: "Registration failed", 
            details: error.message,
            fullError: error
        });
    }
};

// Verify OTP
const verifyOTP = async (req, res) => {
    console.log('OTP verification request received:', req.body);
    const { userId, otp } = req.body;

    try {
        const user = await User.findById(userId);
        if (!user) {
            return res.status(404).json({ error: "User not found" });
        }

        if (user.isVerified) {
            return res.status(400).json({ error: "Email already verified" });
        }

        if (!user.otp || !user.otp.code || isOTPExpired(user.otp.expiresAt)) {
            return res.status(400).json({ error: "OTP expired" });
        }

        if (user.otp.code !== otp) {
            return res.status(400).json({ error: "Invalid OTP" });
        }

        // Verify user and clear OTP
        user.isVerified = true;
        user.otp = undefined;
        
        // Activate 7-day free trial for new users
        if (!user.trialUsed) {
            const now = new Date();
            const trialEndDate = new Date(now);
            trialEndDate.setDate(trialEndDate.getDate() + 7); // 7 days from now
            
            user.trialStartDate = now;
            user.trialEndDate = trialEndDate;
            user.trialUsed = true;
            console.log('7-day free trial activated for user:', user._id, 'Ends on:', trialEndDate);
        }
        
        await user.save();
        console.log('User verified successfully:', user._id);
        // Fire-and-forget WhatsApp welcome after successful verification
        if (WELCOME_MESSAGE_ENABLED && user.phone) {
          try {
            const to = normalizePhone(user.phone);
            if (to && !isPlaceholderPhone(to) && isValidEgyptPhone(to)) {
              const text = buildWelcomeMessage(user);
              if (WELCOME_MESSAGE_DRY_RUN) {
                console.log('[WelcomeMessage] DRY_RUN → would send WhatsApp to', to, { to, text });
              } else {
                // Don't block the response; log errors only
                sendWhatsAppWelcome(to, text).then(() => {
                  console.log('[WelcomeMessage] Sent WhatsApp welcome to', to);
                }).catch(err => {
                  const status = err.response?.status;
                  console.error('[WelcomeMessage] Send error to', to, status ? `HTTP ${status}` : err.message);
                });
              }
            } else {
              console.log('[WelcomeMessage] Skipped: invalid or placeholder phone', user.phone);
            }
          } catch (wmErr) {
            console.error('[WelcomeMessage] Failed to prepare message:', wmErr.message);
          }
        }

        const accessToken = generateAccessToken(user);
        const refreshToken = generateRefreshToken(user);

        res.cookie("refreshToken", refreshToken, {
            httpOnly: true,
            secure: process.env.NODE_ENV === "production",
            sameSite: "Strict",
            maxAge: 30 * 24 * 60 * 60 * 1000,
        });

        res.json({
            message: "Email verified successfully",
            accessToken,
            user: {
                id: user._id,
                name: user.name,
                email: user.email,
                darkMode: user.darkMode,
                role: user.role,
                phone: user.phone
            }
        });
    } catch (error) {
        console.error('OTP verification error:', error);
        res.status(500).json({ 
            error: "Verification failed",
            details: error.message 
        });
    }
};

// Resend OTP
const resendOTP = async (req, res) => {
    console.log('Resend OTP request received:', req.body);
    const { userId } = req.body;

    try {
        const user = await User.findById(userId);
        if (!user) {
            return res.status(404).json({ error: "User not found" });
        }

        if (user.isVerified) {
            return res.status(400).json({ error: "Email already verified" });
        }

        // Generate new OTP
        const otp = generateOTP();
        const otpExpiry = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes

        user.otp = {
            code: otp,
            createdAt: new Date(),
            expiresAt: otpExpiry
        };
        await user.save();
        console.log('New OTP generated for user:', user._id);

        // Send new OTP email
        await sendOTPEmail(user.email, otp);
        console.log('New OTP email sent successfully');

        res.json({ message: "New verification code sent successfully" });
    } catch (error) {
        console.error('Resend OTP error:', error);
        res.status(500).json({ 
            error: "Failed to resend verification code",
            details: error.message 
        });
    }
};

module.exports = {
    register: [otpLimiter, registerUser],
    login: [authLimiter, loginUser],
    refreshToken,
    logoutUser,
    verifyToken,
    updateDarkMode,
    updateUserPhone,
    verifyOTP: [otpLimiter, verifyOTP],
    resendOTP: [otpLimiter, resendOTP]
};
