const Assignment = require('../models/Assignment');
const Course = require('../models/Course');
const CourseSubscription = require('../models/CourseSubscription');

// Create assignment
exports.createAssignment = async (req, res) => {
    try {
        const { title, description, courseId, questions, dueDate, timeLimit, attemptsAllowed } = req.body;
        const userId = req.user._id;

        // Verify course exists and user is instructor
        const course = await Course.findById(courseId);
        if (!course) {
            return res.status(404).json({ error: 'Course not found' });
        }

        if (course.instructor.toString() !== userId.toString() && req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Only course instructor can create assignments' });
        }

        // Validate questions
        if (!questions || questions.length === 0) {
            return res.status(400).json({ error: 'At least one question is required' });
        }

        // Validate MCQ questions have at least 2 choices
        for (const question of questions) {
            if (question.type === 'mcq') {
                if (!question.choices || question.choices.length < 2) {
                    return res.status(400).json({ error: 'MCQ questions must have at least 2 choices' });
                }
                // Ensure at least one choice is correct
                const hasCorrect = question.choices.some(choice => choice.isCorrect);
                if (!hasCorrect) {
                    return res.status(400).json({ error: 'At least one choice must be marked as correct' });
                }
            }
        }

        const assignment = new Assignment({
            title,
            description: description || '',
            course: courseId,
            createdBy: userId,
            questions,
            dueDate: dueDate ? new Date(dueDate) : null,
            timeLimit: timeLimit || null,
            attemptsAllowed: attemptsAllowed || 1,
            status: 'published'
        });

        await assignment.save();

        // Add assignment to course
        course.assignments.push(assignment._id);
        await course.save();

        res.status(201).json(assignment);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Get assignments for a course
exports.getCourseAssignments = async (req, res) => {
    try {
        const { courseId } = req.params;
        const userId = req.user._id;

        // Check if user is subscribed to course
        const subscription = await CourseSubscription.findOne({
            user: userId,
            course: courseId,
            status: 'active'
        });

        const course = await Course.findById(courseId);
        if (!course) {
            return res.status(404).json({ error: 'Course not found' });
        }

        // If user is not instructor/admin and not subscribed, return error
        if (course.instructor.toString() !== userId.toString() && 
            req.user.role !== 'admin' && 
            !subscription) {
            return res.status(403).json({ error: 'You must be subscribed to this course to view assignments' });
        }

        const assignments = await Assignment.find({ course: courseId })
            .populate('createdBy', 'name email')
            .sort({ createdAt: -1 });

        // For students, don't show correct answers
        const isInstructor = course.instructor.toString() === userId.toString() || req.user.role === 'admin';
        
        const assignmentsData = assignments.map(assignment => {
            const assignmentObj = assignment.toObject();
            
            if (!isInstructor) {
                // Hide correct answers for students
                assignmentObj.questions = assignmentObj.questions.map(q => {
                    const question = { ...q };
                    if (question.type === 'mcq') {
                        question.choices = question.choices.map(c => ({
                            text: c.text,
                            _id: c._id
                        }));
                    }
                    delete question.correctAnswer;
                    delete question.answerKey;
                    delete question.explanation;
                    return question;
                });
            }
            
            return assignmentObj;
        });

        res.json(assignmentsData);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Get single assignment
exports.getAssignment = async (req, res) => {
    try {
        const { assignmentId } = req.params;
        const userId = req.user._id;

        const assignment = await Assignment.findById(assignmentId)
            .populate('course', 'title instructor')
            .populate('createdBy', 'name email');

        if (!assignment) {
            return res.status(404).json({ error: 'Assignment not found' });
        }

        // Check subscription
        const subscription = await CourseSubscription.findOne({
            user: userId,
            course: assignment.course._id,
            status: 'active'
        });

        const course = assignment.course;
        const isInstructor = course.instructor.toString() === userId.toString() || req.user.role === 'admin';

        if (!isInstructor && !subscription) {
            return res.status(403).json({ error: 'You must be subscribed to this course' });
        }

        const assignmentObj = assignment.toObject();

        // Hide answers for students
        if (!isInstructor) {
            assignmentObj.questions = assignmentObj.questions.map(q => {
                const question = { ...q };
                if (question.type === 'mcq') {
                    question.choices = question.choices.map(c => ({
                        text: c.text,
                        _id: c._id
                    }));
                }
                delete question.correctAnswer;
                delete question.answerKey;
                delete question.explanation;
                return question;
            });
        }

        res.json(assignmentObj);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Submit assignment
exports.submitAssignment = async (req, res) => {
    try {
        const { assignmentId } = req.params;
        const { answers } = req.body;
        const userId = req.user._id;

        const assignment = await Assignment.findById(assignmentId)
            .populate('course', 'title instructor');

        if (!assignment) {
            return res.status(404).json({ error: 'Assignment not found' });
        }

        // Check subscription
        const subscription = await CourseSubscription.findOne({
            user: userId,
            course: assignment.course._id,
            status: 'active'
        });

        if (!subscription) {
            return res.status(403).json({ error: 'You must be subscribed to this course' });
        }

        // Check attempts
        const userSubmissions = assignment.submissions.filter(
            s => s.student.toString() === userId.toString()
        );
        if (userSubmissions.length >= assignment.attemptsAllowed) {
            return res.status(400).json({ error: 'Maximum attempts reached' });
        }

        // Grade answers
        const gradedAnswers = answers.map(answer => {
            const question = assignment.questions.id(answer.questionId);
            if (!question) return null;

            let isCorrect = false;
            let points = 0;

            if (question.type === 'mcq') {
                const selectedChoice = question.choices.id(answer.answer);
                isCorrect = selectedChoice && selectedChoice.isCorrect;
                points = isCorrect ? question.points : 0;
            } else if (question.type === 'true-false') {
                isCorrect = answer.answer === question.correctAnswer;
                points = isCorrect ? question.points : 0;
            } else {
                // For short-answer and essay, manual grading needed
                // For now, just store the answer
                points = 0;
            }

            return {
                questionId: answer.questionId,
                answer: answer.answer,
                isCorrect,
                points
            };
        }).filter(Boolean);

        const totalScore = gradedAnswers.reduce((sum, a) => sum + a.points, 0);

        // Add submission
        assignment.submissions.push({
            student: userId,
            answers: gradedAnswers,
            score: totalScore,
            attemptNumber: userSubmissions.length + 1
        });

        await assignment.save();

        res.json({
            message: 'Assignment submitted successfully',
            score: totalScore,
            totalPoints: assignment.totalPoints
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Update assignment (instructor only)
exports.updateAssignment = async (req, res) => {
    try {
        const { assignmentId } = req.params;
        const userId = req.user._id;

        const assignment = await Assignment.findById(assignmentId)
            .populate('course', 'instructor');

        if (!assignment) {
            return res.status(404).json({ error: 'Assignment not found' });
        }

        const course = assignment.course;
        if (course.instructor.toString() !== userId.toString() && req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Only instructor can update assignments' });
        }

        Object.assign(assignment, req.body);
        await assignment.save();

        res.json(assignment);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

// Delete assignment
exports.deleteAssignment = async (req, res) => {
    try {
        const { assignmentId } = req.params;
        const userId = req.user._id;

        const assignment = await Assignment.findById(assignmentId)
            .populate('course', 'instructor');

        if (!assignment) {
            return res.status(404).json({ error: 'Assignment not found' });
        }

        const course = assignment.course;
        if (course.instructor.toString() !== userId.toString() && req.user.role !== 'admin') {
            return res.status(403).json({ error: 'Only instructor can delete assignments' });
        }

        // Remove from course
        const courseDoc = await Course.findById(course._id);
        courseDoc.assignments = courseDoc.assignments.filter(
            id => id.toString() !== assignmentId
        );
        await courseDoc.save();

        await Assignment.findByIdAndDelete(assignmentId);

        res.json({ message: 'Assignment deleted successfully' });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};

