'use strict';

const cron = require('node-cron');
const axios = require('axios');
const phoneUtils = require('./utils/phoneUtils');
const groqService = require('./utils/groqService');
const AutomationReport = require('./models/AutomationReport');

// Configurable endpoints and auth
const USER_ACTIVITY_SUMMARY_URL = process.env.USER_ACTIVITY_SUMMARY_URL || 'https://server.task-tree.net/api/user-activity-summary';
const DAILY_SUMMARY_URL = process.env.DAILY_SUMMARY_URL || 'https://server.task-tree.net/api/daily-summary';
const TITI_MESSAGE_URL = process.env.TITI_MESSAGE_URL || 'https://server.task-tree.net/api/titi/generate-message';
const WASENDER_URL = process.env.WASENDER_URL || 'https://wasenderapi.com/api/send-message';
const WASENDER_API_TOKEN = process.env.WASENDER_API_TOKEN || '30ddf4326470551c5ff032627761b0736d5981e304cb1c5a853bbb99fddac45d';

// Scheduling window and controls
const START_TIME = process.env.TITI_AUTOMATION_START || '06:30'; // HH:mm
const END_TIME = process.env.TITI_AUTOMATION_END || '23:00'; // HH:mm
const MIN_TRIGGERS = Number(2);
const MAX_TRIGGERS = Number(3);
const TZ = process.env.TITI_AUTOMATION_TZ; // e.g., 'Africa/Cairo'
const ENABLED = (process.env.TITI_AUTOMATION_ENABLED ?? 'true').toLowerCase() !== 'false';
const DRY_RUN = (process.env.TITI_AUTOMATION_DRY_RUN ?? 'false').toLowerCase() === 'true';
const SOURCE = (process.env.TITI_AUTOMATION_SOURCE || 'daily-summary').toLowerCase(); // 'daily-summary' | 'activity-summary'
const SEND_DELAY_MS = Number(5000);
const QUOTE_RETRY_ATTEMPTS = Number(process.env.TITI_AUTOMATION_QUOTE_RETRY_ATTEMPTS || 3);
const SKIP_INVALID = (process.env.TITI_AUTOMATION_SKIP_INVALID ?? 'true').toLowerCase() === 'true';

// Helper: normalize Egyptian phone to +20XXXXXXXXXX using shared util
function normalizePhone(input) {
  try {
    const normalizer = (typeof phoneUtils === 'function')
      ? phoneUtils
      : (phoneUtils && typeof phoneUtils.normalizeEgyptPhone === 'function')
        ? phoneUtils.normalizeEgyptPhone
        : null;
    return normalizer ? normalizer(input) : String(input || '').trim();
  } catch (_) {
    return String(input || '').trim();
  }
}

// Removed placeholder phone support: no mock numbers are embedded in code.

const logPrefix = '[TitiAutomation]';

let scheduledTasks = [];
let quotesUsedToday = new Set();
let authorsUsedToday = new Set();

function sleep(ms) { return new Promise(res => setTimeout(res, ms)); }

function formatErrorReason(err) {
  try {
    const status = err?.response?.status;
    const body = err?.response?.data;
    const msg = body?.message || body?.error || err?.message || 'Unknown error';
    return status ? `HTTP ${status} - ${String(msg)}` : String(msg);
  } catch (_) {
    return err?.message || 'Unknown error';
  }
}

function parseHHMM(str) {
  const [hh, mm] = String(str).split(':').map(x => parseInt(x, 10));
  return hh * 60 + mm;
}

function formatCronForMinuteOfDay(minuteOfDay) {
  const h = Math.floor(minuteOfDay / 60);
  const m = minuteOfDay % 60;
  return { h, m, cron: `${m} ${h} * * *` };
}

function generateRandomMinutes(count, startMin, endMin) {
  const set = new Set();
  const maxAttempts = 1000;
  let attempts = 0;
  while (set.size < count && attempts < maxAttempts) {
    const rnd = Math.floor(Math.random() * (endMin - startMin + 1)) + startMin;
    set.add(rnd);
    attempts++;
  }
  return Array.from(set).sort((a, b) => a - b);
}

async function fetchRecipientsAndContext() {
  // Targeted test: override recipients with a single phone if provided
  const targetPhoneRaw = process.env.TITI_AUTOMATION_TARGET_PHONE;
  if (targetPhoneRaw && String(targetPhoneRaw).trim()) {
    // Normalize override phone using shared util
    const normalized = normalizePhone(targetPhoneRaw);
    console.log(`${logPrefix} Using TARGET_PHONE override → ${normalized}`);
    return [{ to: normalized }];
  }
  if (SOURCE === 'daily-summary') {
    console.log(`${logPrefix} Fetching daily summary from ${DAILY_SUMMARY_URL} ...`);
    const res = await axios.get(DAILY_SUMMARY_URL, { timeout: 20000 });
    const data = res?.data;
    if (!Array.isArray(data)) {
      throw new Error(`Unexpected daily summary response type: ${typeof data}`);
    }
    console.log(`${logPrefix} Daily summary items: ${data.length}`);
    // Expected shape: [{ to: '+201...', summary: '...' }, ...]
    return data.map(item => ({ to: normalizePhone(item.to), summary: item.summary }));
  }
  console.log(`${logPrefix} Fetching activity summary from ${USER_ACTIVITY_SUMMARY_URL} ...`);
  const res = await axios.get(USER_ACTIVITY_SUMMARY_URL, { timeout: 20000 });
  const data = res?.data;
  if (!Array.isArray(data)) {
    throw new Error(`Unexpected activity summary response type: ${typeof data}`);
  }
  console.log(`${logPrefix} Activity summary items: ${data.length}`);
  // Normalize any 'to' fields present
  return data.map(item => ({ ...item, to: normalizePhone(item.to) }));
}

function extractQuote(text) {
  if (typeof text !== 'string') return null;
  const lines = text.split(/\r?\n/).map(l => l.trim()).filter(Boolean);
  // Try fancy quotes first
  const fancy = text.match(/“([^”]+)”/);
  if (fancy && fancy[1]) return fancy[1].trim();
  // Try plain double quotes
  const plain = text.match(/"([^"]+)"/);
  if (plain && plain[1]) return plain[1].trim();
  // Try single quotes
  const single = text.match(/'([^']+)'/);
  if (single && single[1]) return single[1].trim();
  // Try line before author (— ...)
  const authorIdx = lines.findIndex(l => /^—\s/.test(l) || l.includes('—'));
  if (authorIdx > 0) {
    return lines[authorIdx - 1];
  }
  // Fallback: take the longest line as potential quote if it looks quotable
  const longest = lines.sort((a, b) => b.length - a.length)[0];
  return longest && longest.length > 12 ? longest : null;
}

function extractAuthor(text) {
  if (typeof text !== 'string') return null;
  // Common patterns: “quote” — Author, or "quote" - Author
  const dashMatch = text.match(/—\s*([^\n]+)/);
  if (dashMatch && dashMatch[1]) return dashMatch[1].trim();
  const hyphenMatch = text.match(/-\s*([^\n]+)/);
  if (hyphenMatch && hyphenMatch[1]) return hyphenMatch[1].trim();
  return null;
}

async function generateTitiMessage(user) {
  const useLocal = (process.env.TITI_AUTOMATION_LOCAL_GENERATOR || 'false').toLowerCase() === 'true';
  const useGroq = (process.env.TITI_AUTOMATION_USE_GROQ || 'true').toLowerCase() === 'true';
  const fallbackOnError = (process.env.TITI_AUTOMATION_FALLBACK_ON_ERROR || 'true').toLowerCase() === 'true';

  if (useLocal) {
    return localGenerateMessage(user);
  }

  // Prefer Groq generator when enabled
  if (useGroq) {
    try {
      const text = await groqService.generateMotivation({
        name: user.name,
        activeToday: user.activeToday,
        progressChange: user.progressChange
      });
      return String(text || '').trim();
    } catch (err) {
      const status = err.response?.status;
      console.error(`${logPrefix} Groq generation error for ${user.userId || user.to}:`, status ? `HTTP ${status}` : err.message);
      if (fallbackOnError) {
        return localGenerateMessage(user);
      }
      throw err;
    }
  }

  // Fallback to Titi API generator if Groq not enabled
  const payload = {
    context: {
      name: user.name,
      activeToday: user.activeToday,
      progressChange: user.progressChange,
      tone: 'motivational',
      language: 'arabic',
      includeQuote: true
    },
    instruction: 'اكتب رسالة قصيرة باللهجة المصرية فيها طاقة إيجابية أو حكمة من التاريخ أو مقولة تحفيزية. في النهاية اكتب اسم صاحب المقولة مسبوقًا بشرطة طويلة — اسم المؤلف. نوّع بين المؤلفين (ألبرت أينشتاين، ماري كوري، إسحاق نيوتن، ابن سينا، ابن الهيثم، ابن خلدون، جلال الدين الرومي، هنري فورد، بيتر دركر، توماس إديسون، الخوارزمي، طه حسين...) وتجنب استخدام نابليون بونابرت إلا إذا طُلب صراحة. لو المستخدم غير نشيط اليوم، شجّعه بلطف على مراجعة مهامه.'
  };
  const headers = { 'Content-Type': 'application/json' };
  if (process.env.TITI_API_TOKEN) {
    headers['Authorization'] = `Bearer ${process.env.TITI_API_TOKEN}`;
  }
  try {
    const res = await axios.post(TITI_MESSAGE_URL, payload, { headers, timeout: 20000 });
    const text = typeof res.data === 'string' ? res.data : (res.data?.message || res.data?.text || '');
    return String(text || '').trim();
  } catch (err) {
    const status = err.response?.status;
    console.error(`${logPrefix} Error generating message via Titi API for ${user.userId || user.to}:`, status ? `HTTP ${status}` : err.message);
    if (fallbackOnError) {
      return localGenerateMessage(user);
    }
    throw err;
  }
}

async function getUniqueMessageForUser(user) {
  for (let i = 0; i < QUOTE_RETRY_ATTEMPTS; i++) {
    try {
      const msg = await generateTitiMessage(user);
      const quote = extractQuote(msg);
      const author = extractAuthor(msg);
      if (!quote) {
        // No identifiable quote; avoid repeating author if present
        const authorKey = author ? author.toLowerCase() : null;
        if (authorKey && authorsUsedToday.has(authorKey)) {
          console.log(`${logPrefix} Duplicate author detected without quote, retrying... (${i + 1}/${QUOTE_RETRY_ATTEMPTS})`);
        } else {
          if (authorKey) authorsUsedToday.add(authorKey);
          return msg;
        }
      }
      const quoteKey = quote.toLowerCase();
      const authorKey = author ? author.toLowerCase() : null;
      const quoteDup = quotesUsedToday.has(quoteKey);
      const authorDup = authorKey ? authorsUsedToday.has(authorKey) : false;
      if (!quoteDup && !authorDup) {
        quotesUsedToday.add(quoteKey);
        if (authorKey) authorsUsedToday.add(authorKey);
        return msg;
      }
      console.log(`${logPrefix} Duplicate ${quoteDup ? 'quote' : ''}${quoteDup && authorDup ? ' & ' : ''}${authorDup ? 'author' : ''} detected, retrying... (${i + 1}/${QUOTE_RETRY_ATTEMPTS})`);
    } catch (err) {
      const status = err.response?.status;
      console.error(`${logPrefix} Error generating message for ${user.userId}:`, status ? `HTTP ${status}` : err.message);
      // On error, try again until attempts exhausted
    }
    await sleep(500);
  }
  // Last resort: try once more and accept whatever comes
  try {
    return await generateTitiMessage(user);
  } catch (err) {
    console.error(`${logPrefix} Final attempt failed for ${user.userId}:`, err.message);
    return '';
  }
}

function sanitize(text) {
  return String(text || '').replace(/`/g, '').trim();
}

// Local fallback generator: produces short Arabic messages with a non-repeating quote.
function localGenerateMessage(user) {
  const quotes = [
    { q: 'العقل هو كل شيء، ما تفكر فيه تصبح عليه', a: 'بوذا' },
    { q: 'النجاح هو مجموع جهود صغيرة تُكرر يوميًا', a: 'روبرت كولير' },
    { q: 'لا يمكننا حل مشاكلنا بنفس التفكير الذي استخدمناه عندما أنشأناها', a: 'ألبرت أينشتاين' },
    { q: 'الحياة ليست بحثًا عن الذات، بل صنع الذات', a: 'جورج برنارد شو' },
    { q: 'العلم من دون ضمير خراب الروح', a: 'رابليه' },
    { q: 'إنما تُدرك الحياة بالعمل، لا بالأمنيات', a: 'طه حسين' },
    { q: 'من كان ماضيه شريفًا، كان مستقبله أشرق', a: 'ابن خلدون' },
    { q: 'اعمل بذكاء لا بجهد فقط', a: 'هنري فورد' },
    { q: 'لا شيء يُضيّع الوقت أكثر من انتظار الوقت المناسب', a: 'تشارلز بوكوفسكي' },
    { q: 'عندما تعمل لما تحب، لا تشعر بالتعب', a: 'كونفوشيوس' },
    { q: 'إن لم تستطع فعل أمرٍ عظيم، افعل أشياء صغيرة بشكل عظيم', a: 'نابليون هيل' },
    { q: 'الصبر مفتاح الفرج، والعمل طريق النجاح', a: 'مثل عربي' },
    { q: 'العقل البشري ينمو بالأسئلة أكثر من الإجابات', a: 'ابن الهيثم' },
    { q: 'الفضول هو الوقود الحقيقي للعلم', a: 'ماري كوري' },
    { q: 'الأفكار العظيمة تأتي أثناء العمل لا الانتظار', a: 'توماس إديسون' },
    { q: 'الدواء يداوي الأجسام، والمعرفة تداوي العقول', a: 'ابن سينا' },
    { q: 'ما يُقاس يُمكن تحسينه', a: 'بيتر دركر' },
    { q: 'اتخذ خطوة واحدة اليوم، تعتدل طريقك غدًا', a: 'حكمة' },
    { q: 'لا تتوقف، فالبدايات الصغيرة تصنع نهايات كبيرة', a: 'حكمة' }
  ];
  // Pick a quote not used today if possible
  let picked = null;
  for (const item of quotes) {
    const key = item.q.toLowerCase();
    const authorKey = item.a.toLowerCase();
    if (!quotesUsedToday.has(key) && !authorsUsedToday.has(authorKey)) { picked = item; break; }
  }
  if (!picked) { picked = quotes[Math.floor(Math.random() * quotes.length)]; }

  const baseNeutral = `يا ${user.name || 'صديق تيتي'}، تيتي بيشجعك تكمل وتنجز النهارده ✨`;
  const tail = `\n\n“${picked.q}” — ${picked.a}`;
  const msg = `${baseNeutral}${tail}`;
  // Mark quote as used for today to aid dedup
  quotesUsedToday.add(picked.q.toLowerCase());
  authorsUsedToday.add(picked.a.toLowerCase());
  return msg;
}

// Placeholder phone handling removed: system no longer references test numbers.

function isValidEgyptPhone(to) {
  if (!SKIP_INVALID) return true;
  const digits = String(to).replace(/\D/g, '');
  const msisdn = digits.startsWith('20') ? digits.slice(2) : digits;
  try {
    return !!phoneUtils.isValidEgyptMobile && phoneUtils.isValidEgyptMobile(msisdn);
  } catch (_) {
    // If util not available as named export, assume valid
    return true;
  }
}

async function sendWhatsApp(to, text) {
  const payload = { to, text: sanitize(text) };
  if (DRY_RUN) {
    console.log(`${logPrefix} DRY_RUN → would send WhatsApp to ${to}:`, payload);
    return { dryRun: true, to };
  }
  try {
    const res = await axios.post(WASENDER_URL, payload, {
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${WASENDER_API_TOKEN}`
      },
      timeout: 20000
    });
    console.log(`${logPrefix} Sent WhatsApp to ${to} (status ${res.status}).`);
    return res.data;
  } catch (err) {
    const status = err.response?.status;
    const body = err.response?.data;
    console.error(`${logPrefix} WhatsApp send error to ${to}:`, status ? `HTTP ${status}` : err.message, body ? JSON.stringify(body) : '');
    // Handle rate limiting: retry once after waiting suggested time
    if (status === 429) {
      const retryAfterSec = Number(body?.retry_after) || 5; // default 5s
      const waitMs = Math.max(SEND_DELAY_MS, retryAfterSec * 1000);
      console.log(`${logPrefix} Rate limited. Waiting ${waitMs}ms then retrying to=${to} ...`);
      await sleep(waitMs);
      try {
        const res2 = await axios.post(WASENDER_URL, payload, {
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${WASENDER_API_TOKEN}`
          },
          timeout: 20000
        });
        console.log(`${logPrefix} Sent WhatsApp to ${to} on retry (status ${res2.status}).`);
        return res2.data;
      } catch (err2) {
        const status2 = err2.response?.status;
        const body2 = err2.response?.data;
        console.error(`${logPrefix} Retry failed to ${to}:`, status2 ? `HTTP ${status2}` : err2.message, body2 ? JSON.stringify(body2) : '');
        throw err2;
      }
    }
    throw err;
  }
}

async function sendWithRetries(to, text, maxAttempts = 3) {
  const attemptsLog = [];
  if (DRY_RUN) {
    // Record as dry run without hitting external API
    return { ok: true, attempts: 0, attemptsLog, dryRun: true, data: { dryRun: true, to } };
  }
  for (let attempt = 1; attempt <= maxAttempts; attempt++) {
    try {
      const data = await sendWhatsApp(to, text);
      return { ok: true, attempts: attempt, attemptsLog, data };
    } catch (err) {
      const reason = formatErrorReason(err);
      const code = err?.response?.status || null;
      attemptsLog.push({ attempt, code, reason, timestamp: new Date() });
      // Backoff strategy (respect rate-limit if provided)
      const retryAfterSec = Number(err?.response?.data?.retry_after) || 0;
      const waitMs = retryAfterSec ? Math.max(SEND_DELAY_MS, retryAfterSec * 1000) : Math.max(SEND_DELAY_MS, 1000 * attempt);
      if (attempt < maxAttempts) {
        console.log(`${logPrefix} Attempt ${attempt} failed to=${to}: ${reason}. Waiting ${waitMs}ms before retry...`);
        await sleep(waitMs);
      } else {
        return { ok: false, attempts: attempt, attemptsLog, error: err };
      }
    }
  }
}

async function runTitiEncouragementOnce() {
  const ts = new Date().toISOString();
  console.log(`${logPrefix} Tick started at ${ts}`);
  try {
    const users = await fetchRecipientsAndContext();
    if (!users.length) {
      console.log(`${logPrefix} No users to process.`);
      return { processed: 0 };
    }
    const run = new AutomationReport({
      type: 'titi',
      runId: ts,
      dryRun: DRY_RUN,
      total: users.length,
      startedAt: new Date(ts),
      items: []
    });
    let processed = 0;
    for (let i = 0; i < users.length; i++) {
      const user = users[i];
      // Normalize phone for consistent processing
      const normalizedTo = normalizePhone(user.to);
      if (!normalizedTo) {
        console.log(`${logPrefix} (${i + 1}/${users.length}) Skipping recipient with un-normalizable phone to=${user.to}.`);
        run.items.push({ to: String(user.to || ''), userId: user.userId, message: '', status: 'skipped', attempts: 0, attemptsLog: [{ attempt: 0, code: null, reason: 'Un-normalizable phone', timestamp: new Date() }] });
        continue;
      }
      user.to = normalizedTo;

      if (!isValidEgyptPhone(user.to)) {
        console.log(`${logPrefix} (${i + 1}/${users.length}) Skipping invalid recipient to=${user.to}.`);
        run.items.push({ to: user.to, userId: user.userId, message: '', status: 'skipped', attempts: 0, attemptsLog: [{ attempt: 0, code: null, reason: 'Invalid Egyptian phone', timestamp: new Date() }] });
        continue;
      }
      console.log(`${logPrefix} (${i + 1}/${users.length}) Generating message for to=${user.to}${user.userId ? ` userId=${user.userId}` : ''} ...`);
      try {
        const msg = await getUniqueMessageForUser(user);
        if (!msg) {
          console.warn(`${logPrefix} Empty message for to=${user.to}; skipping send.`);
          run.items.push({ to: user.to, userId: user.userId, message: '', status: 'skipped', attempts: 0, attemptsLog: [{ attempt: 0, code: null, reason: 'Empty message generated', timestamp: new Date() }] });
        } else {
          const result = await sendWithRetries(user.to, msg, 3);
          processed++;
          if (result.ok) {
            run.items.push({ to: user.to, userId: user.userId, message: msg, status: DRY_RUN ? 'dry_run' : 'sent', attempts: result.attempts, attemptsLog: result.attemptsLog, sentAt: new Date() });
            run.successCount = (run.successCount || 0) + 1;
          } else {
            const lastReason = result.attemptsLog?.slice(-1)[0]?.reason || 'Unknown failure';
            run.items.push({ to: user.to, userId: user.userId, message: msg, status: 'failed', attempts: result.attempts, attemptsLog: result.attemptsLog });
            run.failCount = (run.failCount || 0) + 1;
            console.error(`${logPrefix} Final failure to=${user.to}: ${lastReason}`);
          }
          // Log record
          console.log(`${logPrefix} Log → to=${user.to}${user.userId ? ` userId=${user.userId}` : ''} at=${new Date().toISOString()}`);
        }
      } catch (err) {
        // Already logged
        run.items.push({ to: user.to, userId: user.userId, message: '', status: 'failed', attempts: 1, attemptsLog: [{ attempt: 1, code: err?.response?.status || null, reason: formatErrorReason(err), timestamp: new Date() }] });
        run.failCount = (run.failCount || 0) + 1;
      }
      if (i < users.length - 1 && SEND_DELAY_MS > 0) {
        await sleep(SEND_DELAY_MS);
      }
    }
    console.log(`${logPrefix} Tick finished. Processed ${processed}/${users.length}.`);
    run.processed = processed;
    run.endedAt = new Date();
    try {
      await run.save();
      console.log(`${logPrefix} Report saved: type=titi runId=${run.runId} success=${run.successCount || 0} failed=${run.failCount || 0}`);
    } catch (saveErr) {
      console.error(`${logPrefix} Failed to save report:`, saveErr.message);
    }
    return { processed };
  } catch (err) {
    console.error(`${logPrefix} Tick failed:`, err.message);
    return { processed: 0, error: err.message };
  }
}

function clearSchedules() {
  scheduledTasks.forEach(t => {
    try { t.stop(); } catch (_) {}
  });
  scheduledTasks = [];
}

function scheduleForToday() {
  clearSchedules();
  // Reset quotes for the new day
  quotesUsedToday = new Set();
  authorsUsedToday = new Set();

  const startMin = parseHHMM(START_TIME);
  const endMin = parseHHMM(END_TIME);
  const count = Math.max(MIN_TRIGGERS, Math.min(MAX_TRIGGERS, MIN_TRIGGERS + Math.floor(Math.random() * (MAX_TRIGGERS - MIN_TRIGGERS + 1))));
  const minutes = generateRandomMinutes(count, startMin, endMin);
  console.log(`${logPrefix} Scheduling ${minutes.length} random trigger(s) between ${START_TIME} and ${END_TIME}.`);
  minutes.forEach(minuteOfDay => {
    const { h, m, cron: expr } = formatCronForMinuteOfDay(minuteOfDay);
    const options = {};
    if (TZ) options.timezone = TZ;
    const task = cron.schedule(expr, () => {
      const now = new Date();
      console.log(`${logPrefix} Trigger fired at ${now.toLocaleTimeString()} (scheduled ${String(h).padStart(2,'0')}:${String(m).padStart(2,'0')})`);
      runTitiEncouragementOnce();
    }, options);
    scheduledTasks.push(task);
    console.log(`${logPrefix} Scheduled ${String(h).padStart(2,'0')}:${String(m).padStart(2,'0')} (cron: "${expr}")${TZ ? ` TZ=${TZ}` : ''}.`);
  });
}

function startTitiEncouragementAutomation() {
  if (!ENABLED) {
    console.log(`${logPrefix} Disabled via env TITI_AUTOMATION_ENABLED=false`);
    return;
  }
  // Schedule for today immediately
  scheduleForToday();
  // Reset every midnight
  const options = {};
  if (TZ) options.timezone = TZ;
  cron.schedule('0 0 * * *', () => {
    console.log(`${logPrefix} Midnight reset: regenerating schedule and clearing quotes.`);
    scheduleForToday();
  }, options);
  console.log(`${logPrefix} Automation started.`);
}

// Auto-start on import if enabled
try {
  startTitiEncouragementAutomation();
} catch (e) {
  console.error(`${logPrefix} Failed to start automation:`, e.message);
}

module.exports = {
  startTitiEncouragementAutomation,
  runTitiEncouragementOnce
};