const mongoose = require('mongoose');
const dotenv = require('dotenv');
const PricingPlan = require('../models/PricingPlan');
const connectDB = require('../config/db');

dotenv.config();
connectDB();

const pricingPlans = [
    {
        name: 'Free Plan',
        planId: 'free',
        description: 'Basic task management features with limitations',
        monthlyPrice: 0,
        yearlyPrice: 0,
        features: [
            'Create tasks and subtasks',
            'Categories and basic tags',
            'Basic calendar view (no counters)',
            'Personalized task view',
            'Basic daily summaries (in-app only)',
            'Limited workspaces (3 max)',
            'Limited sections per workspace (4 max)',
            'Limited tasks per section (15 max)',
            'Unlimited subtasks',
            'Limited groups (2 max, 3 members each)',
            'No advanced analytics for groups',
            'Limited team collaboration',
            'Limited smart filters',
            'Drag-and-drop tasks',
            'General support only',
            'Limited AI usage (10 queries/day)',
            'Watermarked outputs if relevant',
            'Limited storage',
            'No commercial usage',
            'No advanced analytics',
            'No roles and permissions',
            'No advanced notifications',
            'No premium templates',
            'No priority support',
            'No export options (PDF/SCORM/API)'
        ],
        maxWorkspaces: 3,
        maxTeamMembers: 0,
        active: true
    },
    {
        name: 'Premium Individual',
        planId: 'premium-individual',
        description: 'Full AI-powered features for individuals',
        monthlyPrice: 100,
        yearlyPrice: 1000,
        features: [
            'Unlimited tasks, subtasks, and categories',
            'Unlimited workspaces',
            'Personalized task view (advanced)',
            'Full AI suite: planning, prioritization, summarization',
            'Advanced calendar with live counters',
            'Daily summaries via app, WhatsApp, email',
            'Morning & accountability messages',
            'Roles and permissions',
            'Team assignments & collaboration',
            'Smart filtering and advanced task organization',
            'Advanced analytics (weekly/monthly)',
            'Gantt/Timeline views',
            'Custom dashboards',
            'Unlimited file storage',
            'Premium templates and UI themes',
            'Priority support',
            'Export options: PDF, SCORM, API',
            'Commercial usage allowed',
            'Early access to new features',
            'Premium badges and visual highlights',
            'Premium workspace customization'
        ],
        maxWorkspaces: -1, // Unlimited
        maxTeamMembers: 0,
        active: true
    },
    {
        name: 'Team Plan',
        planId: 'team',
        description: 'Perfect for small teams (1-10 members)',
        monthlyPrice: 300,
        yearlyPrice: 3000,
        features: [
            'All Premium Individual features',
            'Team collaboration',
            'Team assignments',
            'Roles & permissions',
            'Team usage analytics',
            'Workload balancing',
            'Team productivity scoring'
        ],
        maxWorkspaces: -1,
        maxTeamMembers: 10,
        active: true
    },
    {
        name: 'Small Company',
        planId: 'small-company',
        description: 'For companies with 11-30 employees',
        monthlyPrice: 0, // Per employee pricing
        yearlyPrice: 0,
        pricePerEmployee: 150,
        features: [
            'All Team Plan features',
            'Company-wide analytics',
            'Department management',
            'Advanced reporting',
            'Dedicated support'
        ],
        maxWorkspaces: -1,
        maxTeamMembers: -1,
        maxEmployees: 30,
        active: true
    },
    {
        name: 'Medium Company',
        planId: 'medium-company',
        description: 'For companies with 31-100 employees',
        monthlyPrice: 0,
        yearlyPrice: 0,
        pricePerEmployee: 120,
        features: [
            'All Small Company features',
            'Advanced integrations',
            'Custom API access',
            'Priority onboarding',
            'Account manager'
        ],
        maxWorkspaces: -1,
        maxTeamMembers: -1,
        maxEmployees: 100,
        active: true
    },
    {
        name: 'Large Company',
        planId: 'large-company',
        description: 'For companies with 100+ employees',
        monthlyPrice: 0,
        yearlyPrice: 0,
        pricePerEmployee: 80,
        features: [
            'All Medium Company features',
            'Enterprise security',
            'SLA guarantee',
            'Custom development',
            'Dedicated account manager',
            '24/7 support'
        ],
        maxWorkspaces: -1,
        maxTeamMembers: -1,
        maxEmployees: null, // Unlimited
        active: true
    }
];

const initializePricingPlans = async () => {
    try {
        console.log('Initializing pricing plans...');
        
        for (const plan of pricingPlans) {
            const existingPlan = await PricingPlan.findOne({ planId: plan.planId });
            
            if (existingPlan) {
                // Update existing plan
                await PricingPlan.findOneAndUpdate(
                    { planId: plan.planId },
                    { $set: plan },
                    { new: true }
                );
                console.log(`Updated plan: ${plan.name}`);
            } else {
                // Create new plan
                await PricingPlan.create(plan);
                console.log(`Created plan: ${plan.name}`);
            }
        }
        
        console.log('✅ Pricing plans initialized successfully');
        process.exit(0);
    } catch (error) {
        console.error('Error initializing pricing plans:', error);
        process.exit(1);
    }
};

initializePricingPlans();

